package com.edusoho.aiassistant;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;
import android.text.Editable;
import android.text.InputFilter;
import android.text.Spanned;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;

import androidx.activity.result.ActivityResultLauncher;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.recyclerview.widget.LinearLayoutManager;

import com.applandeo.materialcalendarview.CalendarView;
import com.applandeo.materialcalendarview.DatePicker;
import com.applandeo.materialcalendarview.builders.DatePickerBuilder;
import com.applandeo.materialcalendarview.listeners.OnSelectDateListener;
import com.applandeo.materialcalendarview.listeners.OnTodayClickLister;
import com.blankj.utilcode.util.ActivityUtils;
import com.blankj.utilcode.util.ColorUtils;
import com.blankj.utilcode.util.StringUtils;
import com.blankj.utilcode.util.TimeUtils;
import com.blankj.utilcode.util.ToastUtils;
import com.edusoho.aiassistant.adpter.AIStudyScheduleAdapter;
import com.edusoho.aiassistant.databinding.ActivityStudyScheduleBinding;
import com.edusoho.aiassistant.entity.AIStudyScheduleItem;
import com.edusoho.aiassistant.entity.WorkflowRequest;
import com.edusoho.aiassistant.entity.WorkflowResult;
import com.edusoho.aiassistant.http.APIRequest;
import com.edusoho.aiassistant.http.SingleObserver;
import com.edusoho.aiassistant.widget.AIWeekDayWidget;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.TreeSet;

import rx.Observable;
import rx.functions.Func1;

public class StudyScheduleActivity extends AppCompatActivity implements View.OnClickListener {
    private static final String                       COURSE_ID       = "course_id";
    private static final String                       LEARNING_FINISH = "学完即止";
    private static final String                       TAG             = "StudyScheduleActivity";
    private final        APIRequest                   mAPIRequest     = new APIRequest();
    private              List<String>                 mDeadlines      = new ArrayList<>();
    private              String                       mTodayDate      = new SimpleDateFormat("yyyy-MM-dd").format(Calendar.getInstance().getTime());
    private              String                       mStartTime;
    private              String                       mEndTime;
    private              int                          mCourseId;
    private              ActivityStudyScheduleBinding binding;
    private static final int[]                        WEEK_DAYS       = {1, 2, 3, 4, 5, 6, 7};
    private              TreeSet<Integer>             mWeekDays       = new TreeSet<>();
    private              double                       currentTime     = 2.0; // 默认2小时
    private static final double                       TIME_STEP       = 0.5; // 每次增减0.5小时
    private static final double                       MIN_TIME        = 0.5; // 最小时间
    private static final double                       MAX_TIME        = 24.0; // 最大时间
    private              AIStudyScheduleAdapter       mAdapter;

    public static void launch(ActivityResultLauncher<Intent> launcher, int courseId) {
        Activity currActivity = ActivityUtils.getTopActivity();
        Intent intent = new Intent(currActivity, StudyScheduleActivity.class);
        intent.putExtra(COURSE_ID, courseId);
        launcher.launch(intent);
    }

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        binding = ActivityStudyScheduleBinding.inflate(LayoutInflater.from(this));
        setContentView(binding.getRoot());
        changeStatusBar();
        initViews();
        initData();
    }

    @Override
    public void onClick(View v) {
        // 先检查当前点击是否要把最后一个选中的取消
        AIWeekDayWidget widget = ((AIWeekDayWidget) v);
        int selectedCount = 0;
        for (int i = 0; i < binding.llWeekdaySelect.getChildCount(); i++) {
            AIWeekDayWidget w = (AIWeekDayWidget) binding.llWeekdaySelect.getChildAt(i);
            if (w.isSelected()) {
                selectedCount++;
            }
        }

        if (selectedCount == 1 && widget.isSelected()) {
            // 如果这是最后一个选中的，不能取消
            ToastUtils.showShort("必须至少选择一天");
            return;
        }

        // 切换选中状态
        widget.toggleState();
        int weekDay = Integer.parseInt(v.getTag().toString());
        if (v.isSelected()) {
            //下标从 0 开始
            WEEK_DAYS[weekDay - 1] = weekDay;
        } else {
            WEEK_DAYS[weekDay - 1] = 0;
        }
        requestData();
    }

    private void changeStatusBar() {
        Window window = getWindow();
        window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS);
        window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS);
        window.setStatusBarColor(ColorUtils.getColor(android.R.color.white));
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            window.getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR);
        }
    }

    private void initViews() {
        binding.ivBack.setOnClickListener(v -> finish());
        binding.tvStartTime.setOnClickListener(v -> showDateDialog(false, getStartDateListener()));
        binding.tvEndTime.setOnClickListener(v -> {
            if (mDeadlines.isEmpty()) {
                showDateDialog(true, getEndDateListener());
            } else {
                showDeadlinesDialog();
            }
        });
        binding.tvDay1.setOnClickListener(this);
        binding.tvDay2.setOnClickListener(this);
        binding.tvDay3.setOnClickListener(this);
        binding.tvDay4.setOnClickListener(this);
        binding.tvDay5.setOnClickListener(this);
        binding.tvDay6.setOnClickListener(this);
        binding.tvDay7.setOnClickListener(this);

        setupEditText();

        binding.ivIncrease.setOnClickListener(v -> {
            if (currentTime < MAX_TIME) {
                currentTime += TIME_STEP;
                requestData();
                updateTimeDisplay();
            }
        });

        binding.ivDecrease.setOnClickListener(v -> {
            if (currentTime > MIN_TIME) {
                currentTime -= TIME_STEP;
                requestData();
                updateTimeDisplay();
            }
        });

        binding.tvCreate.setOnClickListener(v -> {
            Intent intent = new Intent()
                    .putExtra("startDate", mStartTime)
                    .putExtra("endDate", StringUtils.equals(mEndTime, LEARNING_FINISH) ? null : mEndTime)
                    .putExtra("weekDays", mWeekDays)
                    .putExtra("dailyLearnDuration", String.valueOf(currentTime));
            setResult(Activity.RESULT_OK, intent);
            finish();
        });

        mAdapter = new AIStudyScheduleAdapter();
        binding.rvStudyList.setLayoutManager(new LinearLayoutManager(this));
        binding.rvStudyList.setHasFixedSize(true);
        binding.rvStudyList.setAdapter(mAdapter);
    }

    // 更新时间显示
    private void updateTimeDisplay() {
        // 转换为字符串显示，如果是整数则不显示小数点
        if (currentTime == (int) currentTime) {
            binding.etLearnTime.setText(String.format("%d", (int) currentTime));
        } else {
            binding.etLearnTime.setText(String.format("%.1f", currentTime));
        }
    }

    private void setupEditText() {
        // 限制输入为数字和小数点
        binding.etLearnTime.setInputType(android.text.InputType.TYPE_CLASS_NUMBER | android.text.InputType.TYPE_NUMBER_FLAG_DECIMAL);

        // 添加输入过滤器，限制输入格式
        binding.etLearnTime.setFilters(new InputFilter[]{new InputFilter() {
            @Override
            public CharSequence filter(CharSequence source, int start, int end, Spanned dest, int dstart, int dend) {
                String input = dest.toString() + source.toString();
                if (input.isEmpty()) {
                    return null; // 允许清空
                }
                // 限制只能输入数字和小数点
                if (!input.matches("\\d*\\.?\\d*")) {
                    return "";
                }
                return null;
            }
        }});

        // 添加文本变化监听器，验证输入值
        binding.etLearnTime.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
            }

            @Override
            public void afterTextChanged(Editable s) {
                String input = s.toString();
                if (input.isEmpty()) {
                    return; // 允许临时清空
                }

                try {
                    double value = Double.parseDouble(input);
                    // 检查是否为0.5的倍数
                    if (value % TIME_STEP != 0) {
                        ToastUtils.showShort("请输入0.5的倍数");
                        // 恢复到之前的值
                        updateTimeDisplay();
                        return;
                    }
                    // 检查范围
                    if (value < MIN_TIME || value > MAX_TIME) {
                        ToastUtils.showShort("请输入0.5到24之间的值");
                        updateTimeDisplay();
                        return;
                    }
                    // 更新当前时间
                    currentTime = value;
                } catch (NumberFormatException e) {
                    ToastUtils.showShort("请输入有效的数字");
                    updateTimeDisplay();
                }
            }
        });
    }

    private void initData() {
        for (int weekDay : WEEK_DAYS) {
            mWeekDays.add(weekDay);
        }
        mCourseId = getIntent().getIntExtra(COURSE_ID, 0);
        WorkflowRequest workflowRequest = new WorkflowRequest();
        workflowRequest.setWorkflow(AIConst.WorkFlowType.PLAN_GENERATE_CONFIG, mCourseId);
        mAPIRequest.getWorkflow(workflowRequest)
                .flatMap(new Func1<WorkflowResult, Observable<WorkflowResult>>() {
                    @Override
                    public Observable<WorkflowResult> call(WorkflowResult workflowResult) {
                        mDeadlines = workflowResult.getOutputs().getDeadlines();
                        if (!mDeadlines.isEmpty()) {
                            mDeadlines.add(0, LEARNING_FINISH);
                        }
                        List<Integer> weekDays = new ArrayList<>();
                        for (int weekDay : WEEK_DAYS) {
                            if (weekDay != 0) {
                                weekDays.add(weekDay);
                            }
                        }
                        WorkflowRequest workflowRequest = new WorkflowRequest();
                        workflowRequest.setPreviewPlan(AIConst.WorkFlowType.PLAN_PREVIEW, mCourseId, mStartTime, mEndTime, weekDays, String.valueOf(currentTime));
                        return mAPIRequest.getWorkflow(workflowRequest);
                    }
                })
                .subscribe(new SingleObserver<WorkflowResult>() {
                    @Override
                    public void onNext(WorkflowResult result) {
                        renderList(processScheduleList(result.getOutputs().getTasks()));
                    }
                });

    }

    private void renderList(List<AIStudyScheduleItem> list) {
        if (list == null || list.isEmpty()) {
            mAdapter.getData().clear();
            mAdapter.setEmptyView(R.layout.empty_study_schedule);
            binding.tvCreate.setBackgroundResource(R.drawable.bg_generated_study_schedule_button_disabled);
            binding.tvCreate.setTextColor(ColorUtils.getColor(R.color.color_C0C0C2));
            binding.tvCreate.setEnabled(false);
            return;
        }
        mAdapter.setList(list);
        binding.tvCreate.setBackgroundResource(R.drawable.bg_generated_study_schedule_button);
        binding.tvCreate.setTextColor(ColorUtils.getColor(android.R.color.white));
        binding.tvCreate.setEnabled(true);
    }

    private void showDateDialog(boolean isShowEndTime, OnSelectDateListener listener) {
        try {
            Calendar minDate = Calendar.getInstance();
            minDate.set(Calendar.HOUR_OF_DAY, 0);
            minDate.set(Calendar.MINUTE, 0);
            minDate.set(Calendar.SECOND, 0);
            minDate.set(Calendar.MILLISECOND, 0);

            Calendar maxDate = Calendar.getInstance();
            maxDate.set(Calendar.YEAR, 2025);
            maxDate.set(Calendar.MONTH, Calendar.DECEMBER);
            maxDate.set(Calendar.DAY_OF_MONTH, 31);

            DatePickerBuilder builder = new DatePickerBuilder(this, listener)
                    .todayButtonVisibility(isShowEndTime ? View.VISIBLE : View.GONE)
                    .dialogTodayButtonColor(android.R.color.white)
                    .todayClickListener((dialog) -> {
                        dialog.dismiss();
                        mEndTime = LEARNING_FINISH;
                        binding.tvEndTime.setText(mEndTime);
                        setLearningTimeEveryDay(mEndTime);
                        if (!checkDateWarning(mStartTime, mEndTime)) {
                            requestData();
                        }
                    })
                    .pickerType(CalendarView.ONE_DAY_PICKER)
                    .minimumDate(minDate)
                    .maximumDate(maxDate);

            if (StringUtils.equals(LEARNING_FINISH, binding.tvEndTime.getText().toString())) {
                builder.date(getSelectedDate(mTodayDate));
            } else {
                builder.date(getSelectedDate(binding.tvEndTime.getText().toString()));
            }
            DatePicker datePicker = builder.build();
            datePicker.show();
        } catch (Exception ex) {
            Log.d(TAG, "showDateDialog: " + ex.getMessage());
        }
    }

    private Calendar getSelectedDate(String selectedDate) {
        if (!TextUtils.isEmpty(selectedDate)) {
            try {
                Date date = new SimpleDateFormat("yyyy-MM-dd").parse(selectedDate);
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(date);
                return calendar;
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    private OnSelectDateListener getStartDateListener() {
        return list -> {
            if (!list.isEmpty()) {
                Calendar selectedDate = list.get(0);
                String selectedDateStr = new SimpleDateFormat("yyyy-MM-dd").format(selectedDate.getTime());
                binding.tvStartTime.setText(selectedDateStr);
                mStartTime = selectedDateStr;
                if (!checkDateWarning(mStartTime, mEndTime)) {
                    requestData();
                }
            }
        };
    }

    private OnSelectDateListener getEndDateListener() {
        return list -> {
            Calendar selectedDate = list.get(0);
            String selectedDateStr = new SimpleDateFormat("yyyy-MM-dd").format(selectedDate.getTime());
            binding.tvEndTime.setText(selectedDateStr);
            mEndTime = selectedDateStr;
            setLearningTimeEveryDay(mEndTime);
            if (!checkDateWarning(mStartTime, mEndTime)) {
                requestData();
            }
        };
    }

    private void showDeadlinesDialog() {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setTitle("选择截止日期");

        // 将截止日期列表转换为 String 数组
        String[] deadlineArray = mDeadlines.toArray(new String[0]);

        // 设置单选列表项
        builder.setSingleChoiceItems(deadlineArray, -1, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                binding.tvEndTime.setText(deadlineArray[which]);
                mEndTime = deadlineArray[which];
                setLearningTimeEveryDay(mEndTime);
                if (!checkDateWarning(mStartTime, mEndTime)) {
                    requestData();
                }
                dialog.dismiss();
            }
        });

        // 设置取消按钮
        builder.setNegativeButton("取消", null);

        AlertDialog dialog = builder.create();
        dialog.show();
    }

    private void setLearningTimeEveryDay(String endTime) {
        if (StringUtils.equals(endTime, LEARNING_FINISH)) {
            binding.tvWeekLearnDuration.setVisibility(View.VISIBLE);
            binding.llLearnDuration.setVisibility(View.VISIBLE);
        } else {
            binding.tvWeekLearnDuration.setVisibility(View.GONE);
            binding.llLearnDuration.setVisibility(View.GONE);
        }
    }

    private void requestData() {
        mWeekDays.clear();
        for (int weekDay : WEEK_DAYS) {
            if (weekDay != 0) {
                mWeekDays.add(weekDay);
            }
        }
        WorkflowRequest workflowRequest = new WorkflowRequest();
        if (StringUtils.equals(mEndTime, LEARNING_FINISH)) {
            workflowRequest.setPreviewPlan(AIConst.WorkFlowType.PLAN_PREVIEW, mCourseId, mStartTime, null, new ArrayList<>(mWeekDays), String.valueOf(currentTime));
        } else {
            workflowRequest.setPreviewPlan(AIConst.WorkFlowType.PLAN_PREVIEW, mCourseId, mStartTime, mEndTime, new ArrayList<>(mWeekDays), null);
        }
        mAPIRequest.getWorkflow(workflowRequest)
                .subscribe(new SingleObserver<WorkflowResult>() {
                    @Override
                    public void onNext(WorkflowResult workflowResult) {
                        if (workflowResult != null && workflowResult.getOutputs() != null && workflowResult.getOutputs().getTasks() != null) {
                            renderList(processScheduleList(workflowResult.getOutputs().getTasks()));
                        }
                    }

                    @Override
                    public void onError(Throwable e) {
                        super.onError(e);
                    }
                });
    }

    public List<AIStudyScheduleItem> processScheduleList(List<AIStudyScheduleItem> originalList) {
        // 如果列表为空，直接返回
        if (originalList == null || originalList.isEmpty()) {
            return new ArrayList<>();
        }

        // 1. 获取所有不同的日期，并保持原始顺序
        List<String> uniqueDates = new ArrayList<>();
        Set<String> seenDates = new HashSet<>();
        for (AIStudyScheduleItem item : originalList) {
            String date = item.getDate();
            if (!seenDates.contains(date)) {
                seenDates.add(date);
                uniqueDates.add(date);
            }
        }

        // 2. 创建新的列表，插入日期标题项并设置isLast
        List<AIStudyScheduleItem> resultList = new ArrayList<>();
        for (String date : uniqueDates) {
            // 插入日期标题项 (type=DATE)
            AIStudyScheduleItem dateItem = new AIStudyScheduleItem();
            dateItem.setType(AIStudyScheduleItem.DATE);
            dateItem.setDate(date);
            resultList.add(dateItem);

            // 获取该日期下的所有任务项
            List<AIStudyScheduleItem> dateItems = new ArrayList<>();
            for (AIStudyScheduleItem item : originalList) {
                if (item.getDate().equals(date)) {
                    dateItems.add(item);
                }
            }

            // 设置最后一个ITEM的isLast为true
            for (int i = 0; i < dateItems.size(); i++) {
                if (i == dateItems.size() - 1) {
                    dateItems.get(i).setLast(true); // 最后一个ITEM设为true
                } else {
                    dateItems.get(i).setLast(false); // 其他设为false
                }
                resultList.add(dateItems.get(i));
            }
        }

        return resultList;
    }

    private boolean checkDateWarning(String startTime, String endTime) {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
        try {
            Date startDate = dateFormat.parse(startTime);
            Date endDate = dateFormat.parse(endTime);
            boolean isWarning = startDate.after(endDate);
            if (isWarning) {
                Animation shake = AnimationUtils.loadAnimation(this, R.anim.shake);
                binding.tvDateWarning.setVisibility(View.VISIBLE);
                binding.tvDateWarning.startAnimation(shake);
                binding.tvCreate.setBackgroundResource(R.drawable.bg_generated_study_schedule_button_disabled);
                binding.tvCreate.setTextColor(ColorUtils.getColor(R.color.color_C0C0C2));
                binding.tvCreate.setEnabled(false);
                return true;
            }
            binding.tvCreate.setBackgroundResource(R.drawable.bg_generated_study_schedule_button);
            binding.tvCreate.setTextColor(ColorUtils.getColor(android.R.color.white));
            binding.tvDateWarning.setVisibility(View.GONE);
            binding.tvCreate.setEnabled(true);
        } catch (Exception e) {
            binding.tvDateWarning.setVisibility(View.GONE);
            binding.tvCreate.setBackgroundResource(R.drawable.bg_generated_study_schedule_button);
            binding.tvCreate.setTextColor(ColorUtils.getColor(android.R.color.white));
            binding.tvCreate.setEnabled(true);
            return false;
        }
        binding.tvDateWarning.setVisibility(View.GONE);
        binding.tvCreate.setBackgroundResource(R.drawable.bg_generated_study_schedule_button);
        binding.tvCreate.setTextColor(ColorUtils.getColor(android.R.color.white));
        binding.tvCreate.setEnabled(true);
        return false;
    }
}
