package com.edusoho.aiassistant.widget;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.res.Configuration;
import android.content.res.TypedArray;
import android.graphics.Typeface;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.constraintlayout.widget.ConstraintLayout;
import androidx.fragment.app.FragmentActivity;
import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.LifecycleObserver;
import androidx.lifecycle.Observer;
import androidx.lifecycle.OnLifecycleEvent;
import androidx.lifecycle.ViewModelProvider;

import com.blankj.utilcode.util.ActivityUtils;
import com.blankj.utilcode.util.BarUtils;
import com.blankj.utilcode.util.StringUtils;
import com.edusoho.aiassistant.AIAssistantConfig;
import com.edusoho.aiassistant.AIAssistantSDK;
import com.edusoho.aiassistant.AIConst;
import com.edusoho.aiassistant.AIFloatWidgetButtonListener;
import com.edusoho.aiassistant.AIMessageListActivity;
import com.edusoho.aiassistant.AISDKParams;
import com.edusoho.aiassistant.R;
import com.edusoho.aiassistant.databinding.AiFloatingWidgetLayoutBinding;
import com.edusoho.aiassistant.entity.AISocketMessage;
import com.edusoho.aiassistant.entity.MetaData;
import com.edusoho.aiassistant.util.AIMessageCache;
import com.edusoho.aiassistant.util.AIUnreadMessageCache;
import com.edusoho.aiassistant.viewmodel.AIViewModel;

import java.util.List;

public class AIFloatingWidget extends ConstraintLayout implements LifecycleObserver {
    private static final String                    TAG                      = "AIFloatingWidget";
    private              long                      mDelayHideAvatarDuration = 5000;
    private              AIViewModel               mAIViewModel;
    private              Observer<AISocketMessage> mObserver;
    private              MetaData                  mMetadata                = new MetaData(0, "", 0, "");
    private              String                    mDomainId;
    private              boolean                   isShowAccessible;
    private              boolean                   isGeneratedStudyPlan;
    private              float                     dX, dY;
    private float oX, oY;
    //    private int screenWidth, screenHeight;
    private AiFloatingWidgetLayoutBinding binding;
    private Handler                       handler; // 用于延迟任务
    private Runnable                      hideRunnable; // 隐藏任务
    private String                        mFloatType;
    private boolean                       isAvatarMoveable       = true;  // 头像是否可滑动
    private boolean                       isShowPromptWithAvatar = false; // 头像隐藏的时候，点击头像是否跟随头像一起出来
    private boolean                       isStillShow            = true;  //未读为 0，是否隐藏
    private AIAssistantConfig             mAIAssistantConfig;
    private AIFloatWidgetButtonListener   mAIFloatWidgetButtonListener;

    public AIFloatingWidget(@NonNull Context context) {
        super(context);
        init(context, null);
    }

    public AIFloatingWidget(@NonNull Context context, long delayHideAvatarDuration) {
        super(context);
        this.mDelayHideAvatarDuration = delayHideAvatarDuration;
        init(context, null);
    }

    public AIFloatingWidget(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs);
    }

    public AIFloatingWidget(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context, attrs);
    }

    private void init(Context context, @Nullable AttributeSet attrs) {
        binding = AiFloatingWidgetLayoutBinding.inflate(LayoutInflater.from(context), this);
        handler = new Handler(Looper.getMainLooper()); // 初始化 Handler
        hideRunnable = this::hideHalfBasedOnPosition; // 定义隐藏任务
        if (attrs != null) {
            TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.AIFloatingWidget);
            try {
                // 获取 iconSrc 属性
                int iconResId = typedArray.getResourceId(R.styleable.AIFloatingWidget_iconSrc, R.drawable.ai_icon);
                setIcon(iconResId);
                boolean isShowSayHi = typedArray.getBoolean(R.styleable.AIFloatingWidget_say_hi, false);
                mDelayHideAvatarDuration = typedArray.getInt(R.styleable.AIFloatingWidget_delay_hide, 5000);
                setShowPrompt(isShowSayHi);

                // 获取 messageNum 属性
//                mMessageCount = typedArray.getInt(R.styleable.AIFloatingWidget_messageNum, 0);
//                setMessageNum(mMessageCount);
            } finally {
                typedArray.recycle(); // 回收 TypedArray，避免内存泄漏
            }
        }
        binding.clPrompt.setOnClickListener(null);
        setOnClickListener(v -> {
            setShowPrompt(false);
            clickAvatar();
        });
        handler.postDelayed(hideRunnable, mDelayHideAvatarDuration);
        binding.ivClose.setOnClickListener(v -> setShowPrompt(false));
        binding.tvSendMessageButton.setOnClickListener(v -> {
            setShowPrompt(false);
            setMessageNum(0);
            if (StringUtils.equals(FloatType.COURSE, mFloatType)) {
                AIAssistantSDK.launch(
                        new AIAssistantConfig.Builder()
                                .metaData(mMetadata)
                                .isShowAccessible(isShowAccessible)
                                .isGeneratedStudyPlan(isGeneratedStudyPlan)
                                .domainId(mDomainId)
                                .preMessageType(AIConst.PRE_MESSAGE_TYPE.TEXT)
                                .preSendMessage("制定学习计划")
                                .build()
                );
            } else if (StringUtils.equals(FloatType.QUESTION, mFloatType)) {
                if (mAIFloatWidgetButtonListener != null) {
                    mAIFloatWidgetButtonListener.onButtonClick();
                }
            }
        });
        initViewModel();
    }

    @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
    public void onActivityDestroy() {
        clearViewModel();
    }

    @OnLifecycleEvent(Lifecycle.Event.ON_RESUME)
    public void onActivityResume() {
        setMessageNum(AIUnreadMessageCache.getTotalUnread());
    }

    private void initViewModel() {
        if (mAIViewModel == null) {
            mAIViewModel = new ViewModelProvider(findHostActivity()).get(AIViewModel.class);
        }
        if (mObserver == null) {
            mObserver = aiSocketMessage -> {
                Log.d(TAG, "onChanged: " + aiSocketMessage.toString());
                if (aiSocketMessage.getTotal() > 0) {
                    setVisibility(View.VISIBLE);
                    AIUnreadMessageCache.replace(aiSocketMessage);
                    setMessageNum(AIUnreadMessageCache.getTotalUnread());
                    showAvatar();
                }
            };
        }
        mAIViewModel.getMessageLiveData().observe(findHostActivity(), mObserver);
    }

    private void clearViewModel() {
        if (mAIViewModel != null) {
            mAIViewModel.getMessageLiveData().removeObservers(findHostActivity());
        }
        mAIViewModel = null;
        mObserver = null;
    }

    private void clickAvatar() {
        if (StringUtils.isEmpty(mDomainId)) {
            //mDomainId 为空，是首页进去
            String defaultDomainId = AIUnreadMessageCache.getDefaultDomainId();
//            AIMessageListActivity.launch(mMetadata, defaultDomainId, isShowAccessible);
            AIAssistantSDK.launch(new AIAssistantConfig.Builder()
                    .metaData(mMetadata)
                    .domainId(defaultDomainId)
                    .isShowAccessible(isShowAccessible)
                    .isGeneratedStudyPlan(isGeneratedStudyPlan)
                    .build()
            );
            if (!AIUnreadMessageCache.isAnotherDomainId(defaultDomainId)) {
                setVisibility(INVISIBLE);
            }
        } else {
            AIAssistantSDK.launch(new AIAssistantConfig.Builder()
                    .metaData(mMetadata)
                    .domainId(mDomainId)
                    .isShowAccessible(isShowAccessible)
                    .isGeneratedStudyPlan(isGeneratedStudyPlan)
                    .build()
            );
        }
    }

    public void setStillShow(boolean isStillShow) {
        this.isStillShow = isStillShow;
    }

    /**
     * 显示头像 mDelayHideAvatarDuration 秒后再隐藏
     */
    private void showAvatar() {
        float targetX = getX() < 0 ? 0 : getResources().getDisplayMetrics().widthPixels - getWidth();
        animate()
                .x(targetX)
                .setDuration(300)
                .alpha(1.0f)
                .withEndAction(() -> {
                    handler.postDelayed(hideRunnable, mDelayHideAvatarDuration);
                })
                .start();
    }

    // 获取宿主 Activity（通过 Context 链查找）
    private FragmentActivity findHostActivity() {
        Context context = getContext();
        while (context instanceof ContextWrapper) {
            if (context instanceof FragmentActivity) {
                return (FragmentActivity) context;
            }
            context = ((ContextWrapper) context).getBaseContext();
        }
        throw new IllegalStateException("AIFloatingWidget 必须嵌入 FragmentActivity 中");
    }

    public void setIcon(int iconResId) {
        binding.ivAiAvatar.setImageResource(iconResId);
    }

    public void setAIFloatWidgetButtonListener(AIFloatWidgetButtonListener listener) {
        this.mAIFloatWidgetButtonListener = listener;
    }

    /**
     * 头像上面的欢迎框
     *
     * @param isShow 是否显示
     */
    public void setShowPrompt(boolean isShow) {
        try {
            if (isShow) {
                binding.clPrompt.setVisibility(View.VISIBLE);
                binding.vGap.setVisibility(View.VISIBLE);
            } else {
                Log.d(TAG, "setShowPrompt - binding: " + (binding == null));
                binding.clPrompt.setVisibility(View.GONE);
                binding.vGap.setVisibility(View.GONE);
            }
        } catch (Exception e) {
            Log.d(TAG, e.getMessage());
        }
    }

    /**
     * 头像自动隐藏
     *
     * @param delayMillis 几秒后隐藏，默认 5s
     */
    public void setDelayHideAvatar(long delayMillis) {
        this.mDelayHideAvatarDuration = delayMillis;
    }

    /**
     * 配置头像上面的欢迎框
     *
     * @param title       标题
     * @param content     内容
     * @param buttonText  按钮文案
     * @param isShowClose 是否显示关闭按钮
     */
    public void setPrompt(String title, String content, String buttonText, boolean isShowClose) {
        binding.tvTitle.setText(title);
        binding.tvTitle.setVisibility(StringUtils.isEmpty(title) ? GONE : VISIBLE);
        binding.tvContent.setText(content);
        binding.tvContent.setVisibility(StringUtils.isEmpty(content) ? GONE : VISIBLE);
        binding.tvSendMessageButton.setText(buttonText);
        binding.tvSendMessageButton.setVisibility(StringUtils.isEmpty(buttonText) ? GONE : VISIBLE);
        binding.ivClose.setVisibility(isShowClose ? VISIBLE : GONE);
    }

    /**
     * 消息数量小红点
     *
     * @param num 消息数量
     */
    public void setMessageNum(int num) {
        if (num >= 1) {
            binding.tvMessageNum.setVisibility(View.VISIBLE);
            binding.tvMessageNum.setText(String.valueOf(num));
        } else {
            if (!isStillShow) {
                setVisibility(View.INVISIBLE);
            }
            binding.tvMessageNum.setVisibility(View.GONE);
        }
    }

    public void setAvatarMoveable(boolean isAvatarMoveable) {
        this.isAvatarMoveable = isAvatarMoveable;
    }

    public void setFloatType(String floatType) {
        this.mFloatType = floatType;
    }

    public void setMetadata(MetaData metadata) {
        this.mMetadata = metadata;
    }

    public void setDomainId(String domainId) {
        this.mDomainId = domainId;
    }

    public void setShowPromptWithAvatar(boolean isShow) {
        this.isShowPromptWithAvatar = isShow;
    }

    /**
     * 点击进入聊天框，是否显示功能按钮
     */
    public void setShowAccessible(boolean isShowAccessible) {
        this.isShowAccessible = isShowAccessible;
    }

    public void isGeneratedStudyPlan(boolean isGeneratedStudyPlan) {
        this.isGeneratedStudyPlan = isGeneratedStudyPlan;
    }

    /**
     * 延迟关闭欢迎框
     */
    public void setDelayHidePrompt() {
        handler.postDelayed(() -> setShowPrompt(false), mDelayHideAvatarDuration - 500);
    }

    public void setDelayHide() {
        handler.postDelayed(hideRunnable, mDelayHideAvatarDuration);
    }

    public void setAIAssistantConfig(AIAssistantConfig aiAssistantConfig) {
        this.mAIAssistantConfig = aiAssistantConfig;
    }

    @SuppressLint("ClickableViewAccessibility")
    @Override
    public boolean onTouchEvent(MotionEvent event) {
        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                oX = getX();
                oY = getY();
                dX = getX() - event.getRawX();
                dY = getY() - event.getRawY();
                // 取消之前的隐藏任务
                handler.removeCallbacks(hideRunnable);
                return true;

            case MotionEvent.ACTION_MOVE:
                // 如果处于隐藏状态，不允许拖动
                if (isHalfHidden()) {
                    return true;
                }
                if (!isAvatarMoveable || isPromptVisible()) {
                    return true;
                }
                float newX = event.getRawX() + dX;
                float newY = event.getRawY() + dY;
                newX = Math.max(0, Math.min(newX, getResources().getDisplayMetrics().widthPixels - getWidth()));
                newY = Math.max(0, Math.min(newY, getResources().getDisplayMetrics().heightPixels - getHeight()));
                animate()
                        .x(newX)
                        .y(newY)
                        .setDuration(0)
                        .start();
                return true;

            case MotionEvent.ACTION_UP:
                // 如果处于隐藏状态，点击后完全浮现并重新计时
                if (isHalfHidden()) {
                    float targetX = getX() < 0 ? 0 : getResources().getDisplayMetrics().widthPixels - getWidth();
                    animate()
                            .x(targetX)
                            .setDuration(300)
                            .alpha(1.0f)
                            .withEndAction(() -> {
                                // 浮现后重新启动 5 秒隐藏
                                if (isShowPromptWithAvatar) {
                                    setShowPrompt(true);
                                    setDelayHidePrompt();
                                }
                                handler.postDelayed(hideRunnable, mDelayHideAvatarDuration);
                            })
                            .start();
                    return true;
                }
                // 正常拖动逻辑
                if (Math.abs(oX - getX()) < 10 && Math.abs(oY - getY()) < 10) {
                    performClick();
                } else {
                    // 吸附到最近的边缘
                    float targetX = getX() < (float) getResources().getDisplayMetrics().widthPixels / 2 ? 0 : getResources().getDisplayMetrics().widthPixels - getWidth();
                    animate()
                            .x(targetX)
                            .setDuration(300)
                            .alpha(1.0f)
                            .withEndAction(() -> {
                                // 靠左或靠右时，延迟 5 秒后隐藏一半
                                handler.postDelayed(hideRunnable, mDelayHideAvatarDuration);
                            })
                            .start();
                }
                return true;

            default:
                return false;
        }
    }

    // 根据当前位置隐藏一半
    private void hideHalfBasedOnPosition() {
        if (binding.clPrompt.getVisibility() == View.VISIBLE) return;
        float currentX = getX();
        if (currentX == 0) {
            // 靠左，隐藏一半到左边
            animate()
                    .x(-getWidth() / 2f)
                    .alpha(0.4f)
                    .setDuration(300)
                    .start();
        } else if (currentX == getResources().getDisplayMetrics().widthPixels - getWidth()) {
            // 靠右，隐藏一半到右边
            animate()
                    .x(getResources().getDisplayMetrics().widthPixels - getWidth() / 2f)
                    .setDuration(300)
                    .alpha(0.4f)
                    .start();
        }
    }

    // 判断是否处于半隐藏状态
    private boolean isHalfHidden() {
        float halfHiddenLeft = -getWidth() / 2f;
        float halfHiddenRight = getResources().getDisplayMetrics().widthPixels - getWidth() / 2f;
        float currentX = getX();
        return Math.abs(currentX - halfHiddenLeft) < 1f || Math.abs(currentX - halfHiddenRight) < 1f;
    }

    private boolean isPromptVisible() {
        return binding.clPrompt.getVisibility() == VISIBLE;
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        handler.removeCallbacks(hideRunnable);
        binding = null;
        clearViewModel();
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        // 获取屏幕尺寸
        DisplayMetrics displayMetrics = getResources().getDisplayMetrics();
        int screenWidth = displayMetrics.widthPixels;
        int screenHeight = displayMetrics.heightPixels;

        // 计算可用高度（减去状态栏和ActionBar）
        int usableHeight = screenHeight - BarUtils.getStatusBarHeight() - BarUtils.getActionBarHeight() - (getHeight() / 2);

        Log.d(TAG, "onConfigurationChanged: screen w:" + screenWidth);
        Log.d(TAG, "onConfigurationChanged: screen h:" + screenHeight);
        Log.d(TAG, "Status bar height:" + BarUtils.getStatusBarHeight());
        Log.d(TAG, "ActionBar height:" + BarUtils.getActionBarHeight());
        Log.d(TAG, "Usable height:" + usableHeight);
        Log.d(TAG, "current X:" + getX());
        Log.d(TAG, "current Y:" + getY());

        // 计算控件应该移动到的位置
        float targetX = screenWidth - (getWidth() / 2.0f);  // 右侧
        float targetY = usableHeight - getHeight(); // 可用区域底部

        // 执行动画移动到目标位置
        animate()
                .x(targetX)
                .y(targetY)
                .setDuration(300)
                .start();
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        // 确保控件添加到窗口时重新计算宽度并调整位置
//        getResources().getDisplayMetrics().widthPixels = getResources().getDisplayMetrics().widthPixels;
//        getResources().getDisplayMetrics().heightPixels = getResources().getDisplayMetrics().heightPixels;
        if (!isHalfHidden()) {
            setX(getResources().getDisplayMetrics().widthPixels - getWidth()); // 默认靠右
            handler.postDelayed(hideRunnable, mDelayHideAvatarDuration); // 初始 5 秒后隐藏
        }
    }

    public static final class FloatType {
        public static final String COURSE   = "course";
        public static final String QUESTION = "question";
    }
}
