package com.edusoho.aiassistant.adpter;

import android.os.Build;
import android.text.Spanned;
import android.text.style.AbsoluteSizeSpan;
import android.text.style.LineHeightSpan;
import android.util.Log;
import android.view.View;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.AppCompatImageView;
import androidx.recyclerview.widget.LinearLayoutManager;

import com.blankj.utilcode.util.ConvertUtils;
import com.blankj.utilcode.util.LogUtils;
import com.blankj.utilcode.util.StringUtils;
import com.bumptech.glide.Glide;
import com.chad.library.adapter.base.BaseMultiItemQuickAdapter;
import com.chad.library.adapter.base.BaseQuickAdapter;
import com.chad.library.adapter.base.module.BaseLoadMoreModule;
import com.chad.library.adapter.base.module.BaseUpFetchModule;
import com.chad.library.adapter.base.module.LoadMoreModule;
import com.chad.library.adapter.base.module.UpFetchModule;
import com.chad.library.adapter.base.viewholder.BaseViewHolder;
import com.edusoho.aiassistant.AIConst;
import com.edusoho.aiassistant.AIConst.AIMessageStatus;
import com.edusoho.aiassistant.AIConst.SSEEventType;
import com.edusoho.aiassistant.R;
import com.edusoho.aiassistant.entity.AIMessage;
import com.edusoho.aiassistant.util.KLatexNormalizer;
import com.edusoho.aiassistant.util.MarkwonUtil;
import com.edusoho.aiassistant.widget.AIRatePanel;
import com.edusoho.aiassistant.widget.BottomMarginSpan;

import org.commonmark.node.Heading;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import io.noties.markwon.AbstractMarkwonPlugin;
import io.noties.markwon.Markwon;
import io.noties.markwon.MarkwonConfiguration;
import io.noties.markwon.MarkwonSpansFactory;
import io.noties.markwon.RenderProps;
import io.noties.markwon.SpanFactory;
import io.noties.markwon.core.CoreProps;

public class AIMessageAdapter extends BaseMultiItemQuickAdapter<AIMessage, BaseViewHolder> implements LoadMoreModule, UpFetchModule {

    private static final String TAG = "AIMessageAdapter";

    private Markwon markwonStudyPlan;
    private Markwon markwonNormalStyle;

    public AIMessageAdapter() {
        addItemType(AIMessage.ASSISTANT_LOADING, R.layout.item_loading_message);
        addItemType(AIMessage.ASSISTANT_TEXT, R.layout.item_received_text_message);
        addItemType(AIMessage.ASSISTANT_ANSWER_CARD, R.layout.item_received_answer_card_message);
        addItemType(AIMessage.ASSISTANT_PUSH_CARD, R.layout.item_received_push_card_message);
        addItemType(AIMessage.USER, R.layout.item_send_message);
    }

    public void addSendMessage(AIMessage message) {
        addData(message);
    }

    @Override
    public void addData(@NonNull AIMessage data) {
        super.addData(data);
        LinearLayoutManager layoutManager = (LinearLayoutManager) getRecyclerView().getLayoutManager();
        if (layoutManager != null) {
            layoutManager.setStackFromEnd(true);
            layoutManager.scrollToPositionWithOffset(getData().size() - 1, 0);
        }
    }

    public void addReceivedMessage(AIMessage message) {
        if (StringUtils.equals(message.getEvent(), AIMessageStatus.MESSAGE_LOADING)) {
            addData(message);
        } else if (StringUtils.equals(message.getEvent(), SSEEventType.CHAT_STARTED)) {
            // 如果是chat.created，去找message.loading
            for (int i = getData().size() - 1; i >= 0; i--) {
                AIMessage searchMessage = getData().get(i);
                if (StringUtils.equals(searchMessage.getEvent(), AIMessageStatus.MESSAGE_LOADING)) {
                    searchMessage.setId(message.getId()); //此时 id 实际是chatId
                    searchMessage.setChatId(message.getChatId());
                    searchMessage.setEvent(SSEEventType.CHAT_STARTED); //因为此时 event 是本地设置的 message_loading 占位符。
                    notifyItemChanged(i);
                    return;
                }
            }
        } else if (StringUtils.equals(message.getEvent(), SSEEventType.MESSAGE_DELTA)) {
            for (int i = getData().size() - 1; i >= 0; i--) {
                AIMessage searchMessage = getData().get(i);
                if (StringUtils.equals(searchMessage.getEvent(), SSEEventType.CHAT_STARTED) && StringUtils.equals(searchMessage.getId(), message.getChatId())) {
                    searchMessage.copyFromDeltaMessage(message);
                    Log.i(TAG, "MESSAGE_DELTA - CHAT_CREATED:");
                    notifyItemChanged(i, searchMessage);
                    return;
                } else if (StringUtils.equals(searchMessage.getId(), message.getId())) {
                    searchMessage.copyFromDeltaMessage(message);
                    Log.i(TAG, "MESSAGE_DELTA : getId == getId");
                    notifyItemChanged(i, searchMessage);
                    return;
                }
            }
            addData(message);
        } else if (StringUtils.equals(message.getEvent(), SSEEventType.MESSAGE_COMPLETED)) {
            for (int i = getData().size() - 1; i >= 0; i--) {
                AIMessage searchMessage = getData().get(i);
                if (StringUtils.equals(searchMessage.getEvent(), SSEEventType.CHAT_STARTED) && StringUtils.equals(searchMessage.getId(), message.getChatId())) {
                    //直接从chat.created跳到message.completed，目前只有学习计划生成
                    searchMessage.copyFromCompletedMessage(message);
                    notifyItemChanged(i, searchMessage);
                    return;
                } else if (StringUtils.equals(searchMessage.getId(), message.getId())) {
                    searchMessage.copyFromCompletedMessage(message);
                    Log.i(TAG, "MESSAGE_DELTA : getId == getId");
                    notifyItemChanged(i, searchMessage);
                    return;
                }
            }
            // 找遍整个list没有找到对应的id，属于新消息，那么新增。
            addData(message);
        } else if (StringUtils.equals(message.getEvent(), SSEEventType.CHAT_FAILED)) {
            for (int i = getData().size() - 1; i >= 0; i--) {
                AIMessage searchMessage = getData().get(i);
                if (StringUtils.equals(searchMessage.getEvent(), SSEEventType.CHAT_STARTED) && StringUtils.equals(searchMessage.getId(), message.getChatId())) {
                    //上一条是 chat.created，下一条直接 chat.failed
                    message.setContent("\n\n服务器生成消息失败");
                    searchMessage.copyFromDeltaMessage(message);
                    return;
                } else if (StringUtils.equals(searchMessage.getEvent(), SSEEventType.MESSAGE_DELTA) && StringUtils.equals(searchMessage.getId(), message.getId()) &&
                        StringUtils.equals(searchMessage.getChatId(), message.getChatId())) {
                    message.setContent("\n\n服务器生成消息失败");
                    searchMessage.copyFromDeltaMessage(message);
                    return;
                }
            }
        }
    }

    public void stopAnswer() {
        for (int i = getData().size() - 1; i >= 0; i--) {
            AIMessage targetMessage = getData().get(i);
            if (StringUtils.equals(targetMessage.getRole(), AIMessage.ASSISTANT_ROLE)) {
                if (StringUtils.equals(targetMessage.getEvent(), AIMessageStatus.MESSAGE_LOADING) ||
                        StringUtils.equals(targetMessage.getEvent(), SSEEventType.CHAT_STARTED) ||
                        StringUtils.equals(targetMessage.getEvent(), SSEEventType.MESSAGE_DELTA)) {
                    if (StringUtils.isEmpty(targetMessage.getContentType())) {
                        targetMessage.setContentType(AIConst.ContentType.TEXT);
                    }
                    targetMessage.setEvent(SSEEventType.MESSAGE_COMPLETED);
                    targetMessage.setContent(targetMessage.getTextContent() + "\n\n**已停止回答**");
                    notifyItemChanged(i, targetMessage);
                    break;
                }
            }
        }
    }

    public void updateChatException(String clientId, String exceptionText) {
        try {
            for (int i = getData().size() - 1; i >= 0; i--) {
                AIMessage targetMessage = getData().get(i);
                if (StringUtils.equals(targetMessage.getClientId(), clientId) && StringUtils.equals(targetMessage.getRole(), AIMessage.ASSISTANT_ROLE)) {
                    if (StringUtils.equals(targetMessage.getEvent(), AIMessageStatus.MESSAGE_LOADING) ||
                            StringUtils.equals(targetMessage.getEvent(), SSEEventType.CHAT_STARTED) ||
                            StringUtils.equals(targetMessage.getEvent(), SSEEventType.MESSAGE_DELTA)) {
                        if (StringUtils.isEmpty(targetMessage.getContentType())) {
                            targetMessage.setContentType(AIConst.ContentType.TEXT);
                        }
                        targetMessage.setEvent(SSEEventType.MESSAGE_COMPLETED);
                        targetMessage.setContent(targetMessage.getTextContent() + exceptionText);
                        notifyItemChanged(i, targetMessage);
                        break;
                    }
                }
            }
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
        }
    }

    @Override
    protected void convert(@NonNull BaseViewHolder holder, AIMessage message, @NonNull List<?> payloads) {
        if (payloads.isEmpty()) {
            convert(holder, message);
        } else if (holder.getItemViewType() == AIMessage.ASSISTANT_TEXT) {
            for (Object payload : payloads) {
                if (payload instanceof AIMessage) {
                    AIMessage item = (AIMessage) payload;
                    renderReceiveMarkdown(holder.getView(R.id.tv_content), item.getTextContent(), StringUtils.equals(SSEEventType.MESSAGE_COMPLETED, message.getEvent()));
                    if (StringUtils.equals(item.getEvent(), SSEEventType.MESSAGE_COMPLETED) || item.isFromList()) {
                        holder.setVisible(R.id.ll_function_panel, true);
                        AIRatePanel aiRatePanel = holder.getView(R.id.ai_rate_penal);
                        renderFunctionPanel(aiRatePanel, item);
                    } else {
                        holder.setGone(R.id.ll_function_panel, true);
                    }
                }
            }
        }
    }

    @Override
    protected void convert(@NonNull BaseViewHolder holder, AIMessage message) {
        if (holder.getItemViewType() == AIMessage.ASSISTANT_TEXT) {
            renderReceiveMarkdown(holder.getView(R.id.tv_content), message.getTextContent(), StringUtils.equals(SSEEventType.MESSAGE_COMPLETED, message.getEvent()));
            if (StringUtils.equals(message.getEvent(), SSEEventType.MESSAGE_COMPLETED) || message.isFromList()) {
                holder.setVisible(R.id.ll_function_panel, true);
                AIRatePanel aiRatePanel = holder.getView(R.id.ai_rate_penal);
                renderFunctionPanel(aiRatePanel, message);
            } else {
                holder.setGone(R.id.ll_function_panel, true);
            }
        } else if (holder.getItemViewType() == AIMessage.ASSISTANT_ANSWER_CARD) {
            holder.setText(R.id.tv_card_title, message.getCardContent().getTitle());
            holder.setText(R.id.tv_card_content, message.getCardContent().getDescription());
        } else if (holder.getItemViewType() == AIMessage.ASSISTANT_PUSH_CARD) {
            holder.setText(R.id.tv_card_title, message.getCardContent().getTitle());
            holder.setText(R.id.tv_card_content, message.getCardContent().getDescription());
            AppCompatImageView imageView = holder.getView(R.id.iv_image);
            if (StringUtils.isEmpty(message.getCardContent().getImage())) {
                imageView.setVisibility(View.GONE);
            } else {
                imageView.setVisibility(View.VISIBLE);
                Glide.with(getContext()).load(message.getCardContent().getImage()).into(imageView);
            }
        } else if (holder.getItemViewType() == AIMessage.USER) {
            renderUserMarkdown(holder.getView(R.id.tv_content), message.getTextContent());
        } else if (holder.getItemViewType() == AIMessage.ASSISTANT_LOADING) {
            holder.getView(R.id.iv_audio_animation).setPadding(-120, -120, -120, -120);
        }
    }

    private void renderFunctionPanel(AIRatePanel aiRatePanel, AIMessage message) {
        String rate = StringUtils.isEmpty(message.getRating()) ? "" : message.getRating();
        switch (rate) {
            case "good":
                aiRatePanel.setRate(1);
                break;
            case "bad":
                aiRatePanel.setRate(2);
                break;
            default:
                aiRatePanel.setRate(0);
        }
    }

    @NonNull
    @Override
    public BaseLoadMoreModule addLoadMoreModule(@NonNull BaseQuickAdapter<?, ?> baseQuickAdapter) {
        return new BaseLoadMoreModule(this);
    }

    @NonNull
    @Override
    public BaseUpFetchModule addUpFetchModule(@NonNull BaseQuickAdapter<?, ?> baseQuickAdapter) {
        return new BaseUpFetchModule(this);
    }

    private void renderUserMarkdown(TextView textView, String content) {
        if (content == null) {
            textView.setText("");
        } else {
            final Markwon markwon = Markwon.builder(getContext())
                    .usePlugin(new AbstractMarkwonPlugin() {
                        @Override
                        public void configureSpansFactory(@NonNull MarkwonSpansFactory.Builder builder) {
                            builder.appendFactory(Heading.class, new SpanFactory() {

                                @Nullable
                                @Override
                                public Object getSpans(@NonNull MarkwonConfiguration configuration, @NonNull RenderProps props) {
                                    Integer level = CoreProps.HEADING_LEVEL.get(props);
                                    if (level == 3) {
                                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                                            return new Object[]{new AbsoluteSizeSpan(ConvertUtils.dp2px(16)),
                                                    new LineHeightSpan.Standard(10),
                                                    new BottomMarginSpan(-38)};
                                        }
                                    }
                                    return null;
                                }
                            });
                        }
                    }).build();
            markwon.setMarkdown(textView, content);
        }
    }

    private void renderReceiveMarkdown(TextView textView, String content, boolean isComplete) {
        if (content == null) {
            textView.setText("");
        } else {
            if (isComplete || content.contains("链接直达任务") || isCourseLessonLink(content)) {
                if (markwonStudyPlan == null) {
                    markwonStudyPlan = MarkwonUtil.createStudyPlanStyle(getContext());
                }
                markwonStudyPlan.setMarkdown(textView, content);
            } else {
                if (markwonNormalStyle == null) {
                    markwonNormalStyle = MarkwonUtil.createNormalStyle(getContext(), textView);
                }
                String it = KLatexNormalizer.INSTANCE.normalizeLatex(content);
                Spanned spanned = markwonNormalStyle.render(markwonNormalStyle.parse(it));
//                LogUtils.d(spanned.toString());
                markwonNormalStyle.setParsedMarkdown(textView, spanned);
            }
        }
    }

    private boolean isCourseLessonLink(String link) {
        String regex = "course/\\d+/task/\\d+";
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(link);
        return matcher.find();
    }
}
