package com.edusoho.aiassistant.util

import android.util.Log

object KLatexNormalizer {
    private val INLINE_LATEX = Regex("""\\\((.+?)\\\)""") // \( ... \)
    private val BLOCK_LATEX = Regex("""\\\[(.+?)\\\]""", setOf(RegexOption.DOT_MATCHES_ALL)) // \[ ... \]
    private val FENCED_LATEX = Regex("```latex([\\s\\S]+?)```", RegexOption.MULTILINE)
    private val INLINE_DOLLAR = Regex("""\$(?!\$)([^$\n]*?)\$(?!\$)""")
    private val SINGLELINE_DOUBLE_DOLLAR = Regex("""\$\$(?!\s*\n)(.+?)(?<!\n)\$\$""")

    public fun normalizeLatex(raw: String?): String {
        if (raw.isNullOrEmpty()) return ""

        return try {
            // Phase A: 先在原文里为括号/方括号风格加内边距空格
            var text = INLINE_LATEX.replace(raw) { mr ->
                "\\( ${mr.groupValues[1].trim()} \\)"
            }
            text = BLOCK_LATEX.replace(text) { mr ->
                "\\[ ${mr.groupValues[1].trim()} \\]"
            }

            // Phase B: 处理 ```latex ... ``` 块，生成 $$ 包裹（避免二次嵌套）
            var normalized = FENCED_LATEX.replace(text) { mr ->
                val content = mr.groupValues[1].trim()
                // 去掉一层可能存在的 \[ ... \] 外围，仅保留中间内容
                val innerNoBracket = content
                    .replace(Regex("""^\s*\\\\\[\s*\n?"""), "")
                    .replace(Regex("""\n?\s*\\\\\]\s*$"""), "")
                val innerInlineSpaced = INLINE_LATEX.replace(innerNoBracket) { m ->
                    "\\( ${m.groupValues[1].trim()} \\)"
                }
                "\n$$\n${innerInlineSpaced.trim()}\n$$\n"
            }

            // Phase C: 将剩余括号/方括号风格统一成 $ / $$
            normalized = INLINE_LATEX.replace(normalized) { mr ->
                "${'$'}${mr.groupValues[1]}${'$'}"
            }
            normalized = BLOCK_LATEX.replace(normalized) { mr ->
                "${'$'}${'$'}${mr.groupValues[1]}${'$'}${'$'}"
            }

            // Phase D: 为 $...$ 与 单行 $$...$$ 加入内边距空格（标准化为 `$ x $` / `$$ x $$`）
            normalized = INLINE_DOLLAR.replace(normalized) { mr ->
                "${'$'} ${mr.groupValues[1].trim()} ${'$'}"
            }
            normalized = SINGLELINE_DOUBLE_DOLLAR.replace(normalized) { mr ->
                "${'$'}${'$'} ${mr.groupValues[1].trim()} ${'$'}${'$'}"
            }

            // Phase E: 清理紧邻的空 $$ 围栏，避免空块
            normalized = normalized
                .replace(Regex("""(?m)^\$\$\s*\n\$\$\s*$"""), "$$")
                .replace(Regex("""(?m)^\$\$\s*\n\s*\n\$\$\s*$"""), "$$")

            Log.i("SSE", "normalizeLatex: $normalized")
            normalized
        } catch (ex: Exception) {
            Log.e("SSE", "normalizeLatex error: ${ex.message}")
            raw
        }
    }
}