package com.edusoho.aiassistant;

import android.app.Activity;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.RelativeLayout;

import androidx.activity.result.ActivityResult;
import androidx.activity.result.ActivityResultCallback;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.lifecycle.Observer;
import androidx.lifecycle.ViewModelProvider;
import androidx.recyclerview.widget.LinearLayoutManager;

import com.blankj.utilcode.util.ActivityUtils;
import com.blankj.utilcode.util.ClipboardUtils;
import com.blankj.utilcode.util.ColorUtils;
import com.blankj.utilcode.util.ConvertUtils;
import com.blankj.utilcode.util.GsonUtils;
import com.blankj.utilcode.util.NetworkUtils;
import com.blankj.utilcode.util.StringUtils;
import com.blankj.utilcode.util.ToastUtils;
import com.chad.library.adapter.base.listener.OnUpFetchListener;
import com.edusoho.aiassistant.adpter.AIMessageAdapter;
import com.edusoho.aiassistant.databinding.ActivityAiMessageListBinding;
import com.edusoho.aiassistant.entity.AIFeedback;
import com.edusoho.aiassistant.entity.AIMessage;
import com.edusoho.aiassistant.entity.AISendMessage;
import com.edusoho.aiassistant.entity.AISocketMessage;
import com.edusoho.aiassistant.entity.AudioTokenData;
import com.edusoho.aiassistant.entity.WorkflowRequest;
import com.edusoho.aiassistant.http.APIRequest;
import com.edusoho.aiassistant.http.SSEClient;
import com.edusoho.aiassistant.http.SingleObserver;
import com.edusoho.aiassistant.speech.AsrUtil;
import com.edusoho.aiassistant.speech.BytedanceSpeechEngineAdapter;
import com.edusoho.aiassistant.speech.FunAsrEngineAdapter;
import com.edusoho.aiassistant.speech.SpeechEngineAdapter;
import com.edusoho.aiassistant.speech.SpeechEngineSDK;
import com.edusoho.aiassistant.util.AIMessageCache;
import com.edusoho.aiassistant.util.AIUnreadMessageCache;
import com.edusoho.aiassistant.util.AIUtil;
import com.edusoho.aiassistant.util.MessageDeserializer;
import com.edusoho.aiassistant.viewmodel.AIViewModel;
import com.edusoho.aiassistant.widget.AIRatePanel;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import java.net.SocketException;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeSet;

import okhttp3.Response;
import okhttp3.sse.EventSource;
import rx.Observable;

public class AIMessageListActivity extends AppCompatActivity {
    private static final String                       TAG               = "AIMessageListActivity";
    private static final int                          AI_FEEDBACK_DELAY = 10 * 1000;
    private final        Handler                      mUIHandler        = new Handler(Looper.getMainLooper());
    private final        APIRequest                   mAPIRequest       = new APIRequest();
    private              ActivityAiMessageListBinding binding;
    private              AIMessageAdapter             mMessageAdapter;
    private              LinearLayoutManager          mLayoutManager;
    private              AIViewModel                  mAIViewModel;
    private              Observer<AISocketMessage>    mObserver;

    private AIAssistantConfig mConfig;

    private ActivityResultLauncher<Intent> launcher;
    private SpeechEngineAdapter            mSpeechEngineAdapter;

    public static void launch(AIAssistantConfig config) {
        Activity currActivity = ActivityUtils.getTopActivity();
        Intent intent = new Intent(currActivity, AIMessageListActivity.class);
        intent.putExtra(AIConst.AI_CONFIG, config);
        ActivityUtils.startActivity(intent);
    }

    private void changeStatusBar() {
        Window window = getWindow();
        window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS);
        window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS);
        window.setStatusBarColor(ColorUtils.getColor(R.color.color_F5F5F7));
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            window.getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR);
        }
    }

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        binding = ActivityAiMessageListBinding.inflate(LayoutInflater.from(this));
        setContentView(binding.getRoot());
        changeStatusBar();
        launcher = registerForActivityResult(new ActivityResultContracts.StartActivityForResult(), new ActivityResultCallback<ActivityResult>() {
            @Override
            public void onActivityResult(ActivityResult result) {
                Intent data = result.getData();
                if (data != null) {
                    String startDate = data.getStringExtra("startDate");
                    String endDate = data.getStringExtra("endDate");
                    TreeSet<Integer> weekDays = (TreeSet) data.getSerializableExtra("weekDays");
                    String dailyLearnDuration = data.getStringExtra("dailyLearnDuration");
                    List<Integer> list = new ArrayList<>();
                    if (weekDays != null) {
                        list.addAll(weekDays);
                    }
                    String content = AIUtil.formatStudyTime1(startDate, endDate, list, dailyLearnDuration);
                    AISendMessage aiSendMessage = AIUtil.createGeneratedPlanMessage(startDate, endDate, dailyLearnDuration, list, content, mConfig.getDomainId(), mConfig.getMetaData());
                    sendMessage(aiSendMessage);
                }
            }
        });
        initIntentData();
        initViews();
        initData();
        initSpeechEngine();
        initViewModel();
    }

    @Override
    protected void onDestroy() {
        AsrUtil.releaseEngine();
        super.onDestroy();
        mUIHandler.removeCallbacksAndMessages(null);
        clearViewModel();
    }

    private AIEventSourceListener createEventSourceListener(String uuid) {
        return new AIEventSourceListener(uuid) {
            @Override
            public void onOpen(@NonNull EventSource eventSource, @NonNull Response response) {
                Log.i(TAG, "Connection opened");
            }

            @Override
            public void onEvent(@NonNull EventSource eventSource, String id, String type, @NonNull String data) {
                Log.i(TAG, "Received event: ID=" + id + ", Type=" + type + ", Data=" + data);
                switch (type) {
                    case AIConst.SSEEventType.CHAT_STARTED:
                        AIMessage chatCreatedMessage = GsonUtils.fromJson(data, AIMessage.class);
                        chatCreatedMessage.setRole(AIMessage.ASSISTANT_ROLE);
                        chatCreatedMessage.setEvent(type);
                        mUIHandler.post(() -> mMessageAdapter.addReceivedMessage(chatCreatedMessage));
                        Log.i(TAG, "onEvent chat.created: " + chatCreatedMessage);
                        break;
                    case AIConst.SSEEventType.MESSAGE_DELTA:
                        AIMessage deltaMessage = GsonUtils.fromJson(data, AIMessage.class);
                        deltaMessage.setEvent(type);
//                        mUIHandler.post(() -> {
//                            View view = binding.rvMessageList.getLayoutManager().findViewByPosition(mMessageAdapter.getData().size() - 1);
//                            TextView textView = view.findViewById(R.id.tv_content);
//                            if (textView != null) {
//                                textView.setText(textView.getText().toString() + deltaMessage.getTextContent());
//                            }
//                        });
                        mUIHandler.post(() -> mMessageAdapter.addReceivedMessage(deltaMessage));
                        Log.i(TAG, "onEvent message. delta: " + deltaMessage);
                        break;
                    case AIConst.SSEEventType.MESSAGE_COMPLETED:
                        Gson gson = new GsonBuilder()
                                .registerTypeAdapter(AIMessage.class, new MessageDeserializer())
                                .create();
                        AIMessage completeMessage = GsonUtils.fromJson(gson, data, AIMessage.class);
                        if (StringUtils.equals(AIConst.MessageType.FEEDBACK, completeMessage.getType())) {
                            mUIHandler.postDelayed(() -> showFeedback(completeMessage.getChatId()), AI_FEEDBACK_DELAY);
                        } else if (StringUtils.equals(AIConst.MessageType.ANSWER, completeMessage.getType())) {
                            completeMessage.setEvent(type);
                            AIMessageCache.add(mConfig.getDomainId(), completeMessage);
                            mUIHandler.post(() -> mMessageAdapter.addReceivedMessage(completeMessage));
                        }
                        Log.i(TAG, "onEvent message. completed: " + completeMessage);
                        break;
                    case AIConst.SSEEventType.CHAT_FAILED:
                        AIMessage failedMessage = GsonUtils.fromJson(data, AIMessage.class);
                        failedMessage.setEvent(type);
                        mUIHandler.post(() -> mMessageAdapter.addReceivedMessage(failedMessage));
                        Log.i(TAG, "onEvent message. failed: " + failedMessage);
                        break;
                    case AIConst.SSEEventType.CHAT_COMPLETED:
                        Log.i(TAG, "onEvent message. done. ");
                        mUIHandler.post(() -> binding.aiMessageBox.setTextInputMode());
                        break;
                }
            }

            @Override
            public void onClosed(@NonNull EventSource eventSource) {
                Log.i(TAG, "Connection closed");
            }

            @Override
            public void onFailure(@NonNull EventSource eventSource, Throwable t, Response response) {
                try {
                    if (!NetworkUtils.isConnected()) {
                        mUIHandler.post(() -> mMessageAdapter.updateChatException(mClientId, "\n\n**无法连接到网络**"));
                        Log.i(TAG, "-网络无法连接");
                    } else if (t instanceof SocketException) {
                        Log.i(TAG, "-断开连接");
                        mUIHandler.post(() -> mMessageAdapter.updateChatException(mClientId, "\n\n**网络连接已断开**"));
                    }
                    mUIHandler.post(() -> binding.aiMessageBox.setTextInputMode());
                } catch (Exception e) {
                    Log.e(TAG, e.getMessage());
                }
            }
        };
    }

    private void initIntentData() {
        mConfig = (AIAssistantConfig) getIntent().getSerializableExtra(AIConst.AI_CONFIG);
    }

    private void initViews() {
        mMessageAdapter = new AIMessageAdapter();
        mMessageAdapter.addChildClickViewIds(R.id.iv_like, R.id.iv_dislike, R.id.iv_clipboard);
        mMessageAdapter.setOnItemClickListener((adapter, view, position) -> {
            try {
                AIMessage message = ((AIMessageAdapter) adapter).getItem(position);
                if (message.getItemType() == AIMessage.ASSISTANT_ANSWER_CARD) {
                    StudyScheduleActivity.launch(launcher, mConfig.getMetaData().getCourseId());
                } else if (message.getItemType() == AIMessage.ASSISTANT_PUSH_CARD) {
                    AIUtil.openNews(message.getCardContent().getData().getUrl());
                }
            } catch (Exception e) {
                Log.d(TAG, "click card :" + e.getMessage());
            }
        });
        mMessageAdapter.setOnItemChildClickListener((adapter, view, position) -> {
            AIMessage message = ((AIMessageAdapter) adapter).getItem(position);
            if (view.getId() == R.id.iv_like) {
                pressRating((AIRatePanel) view.getParent(), message, "good");
            } else if (view.getId() == R.id.iv_dislike) {
                pressRating((AIRatePanel) view.getParent(), message, "bad");
            } else if (view.getId() == R.id.iv_clipboard) {
                ClipboardUtils.copyText(message.getTextContent());
                ToastUtils.showShort("复制成功");
            }
        });
        mMessageAdapter.getUpFetchModule().setUpFetchEnable(true);
        mMessageAdapter.getUpFetchModule().setOnUpFetchListener(new OnUpFetchListener() {
            @Override
            public void onUpFetch() {
                mMessageAdapter.getUpFetchModule().setUpFetching(true);
                String beforeMessageId = mMessageAdapter.getData().get(0).getId();
                getNetMessageList(mConfig.getDomainId(), beforeMessageId);
            }
        });
        mLayoutManager = new LinearLayoutManager(this);
        binding.rvMessageList.setLayoutManager(mLayoutManager);
        binding.rvMessageList.setAdapter(mMessageAdapter);
        binding.ivClose.setOnClickListener(v -> finish());
        binding.aiMessageBox.setListener((msg, audio, audioDuration) -> {
            binding.aiMessageBox.stopAnswer();
            mMessageAdapter.stopAnswer();
            if (audio) {
                sendMessage(AIUtil.createAudioTextMessage(msg, mConfig.getDomainId(), true, audioDuration, mConfig.getMetaData()));
            } else {
                sendMessage(AIUtil.createTextMessage(msg, mConfig.getDomainId(), mConfig.getMetaData()));
            }
        });
        initAccessibleLayout();
    }

    private void initAccessibleLayout() {
        binding.btnGeneratePlan.setOnClickListener(v -> {
            binding.aiMessageBox.stopAnswer();
            mMessageAdapter.stopAnswer();
            String msg = binding.btnGeneratePlan.getText();
            sendMessage(AIUtil.createTextMessage(msg, mConfig.getDomainId(), mConfig.getMetaData()));
        });
        binding.btnCheckPlan.setOnClickListener(v -> {
            binding.aiMessageBox.stopAnswer();
            mMessageAdapter.stopAnswer();
            String msg = binding.btnCheckPlan.getText();
            sendMessage(AIUtil.createTextMessage(msg, mConfig.getDomainId(), mConfig.getMetaData()));
        });
        binding.btnRegeneratePlan.setOnClickListener(v -> {
            binding.aiMessageBox.stopAnswer();
            mMessageAdapter.stopAnswer();
            String msg = binding.btnRegeneratePlan.getText();
            sendMessage(AIUtil.createTextMessage(msg, mConfig.getDomainId(), mConfig.getMetaData()));
        });
        if (mConfig.isShowAccessible()) {
            if (mConfig.isGeneratedStudyPlan()) {
                binding.btnGeneratePlan.setVisibility(View.GONE);
                binding.btnCheckPlan.setVisibility(View.VISIBLE);
                binding.btnRegeneratePlan.setVisibility(View.VISIBLE);
            } else {
                binding.btnGeneratePlan.setVisibility(View.VISIBLE);
                binding.btnCheckPlan.setVisibility(View.GONE);
                binding.btnRegeneratePlan.setVisibility(View.GONE);
            }
        } else {
            binding.llAccessibleLayout.setVisibility(View.GONE);
        }
    }

    private void sendMessage(AISendMessage aiSendMessage) {
        //生成一条用户发送的Message
        AIMessage userSendMessage = createUserSendMessage(aiSendMessage.getContent());
        //添加到缓存
        AIMessageCache.add(mConfig.getDomainId(), userSendMessage);
        //显示在 UI MessageList 上
        mUIHandler.post(() -> mMessageAdapter.addSendMessage(userSendMessage));
        //生成一条 loading 的 Message
        final AIMessage placeholderMessage = createPlaceholderMessage();
        placeholderMessage.setEvent(AIConst.AIMessageStatus.MESSAGE_LOADING);
        //loading Message 在 UI MessageList 上显示
        mUIHandler.post(() -> mMessageAdapter.addReceivedMessage(placeholderMessage));
        //户发送的Message 通过 SSE 发送
        final SSEClient sseClient = new SSEClient();
        sseClient.send(aiSendMessage, createEventSourceListener(placeholderMessage.getClientId()));
        binding.aiMessageBox.setCurrentSSEClient(sseClient);
    }

    private void pressRating(AIRatePanel aiRatePanel, AIMessage message, String rating) {
        mAPIRequest.feedback(new AIFeedback(message.getId(), rating))
                .subscribe(new SingleObserver<Boolean>() {
                    @Override
                    public void onNext(Boolean result) {
                        if (result) {
                            ToastUtils.showShort("操作成功");
                            aiRatePanel.setRate(StringUtils.equals("good", rating) ? 1 : 2);
                            message.setRating(rating);
                        }
                    }

                    @Override
                    public void onError(Throwable e) {
                        ToastUtils.showShort(e.getMessage());
                    }
                });
    }

    private void initData() {
        if (AIUnreadMessageCache.getTotalUnread() > 0) {
            mAPIRequest.getPullUnread(mConfig.getDomainId()).subscribe(new SingleObserver<List<AIMessage>>() {
                @Override
                public void onNext(List<AIMessage> list) {
                    AIUnreadMessageCache.removeUnread(mConfig.getDomainId());
                }
            });
        }
        Observable<List<AIMessage>> observable = mAPIRequest.getMessageListFromNet(mConfig.getDomainId());
//        if (!AIMessageCache.get(mDomainId).isEmpty() && AIUnreadMessageCache.getTotalUnread() > 0) {
//            //如果 cache 不为空，有 unread
//            observable = mAPIRequest.getMessageListByIds(mDomainId, AIUtil.convertMessagesToString(AIUnreadMessageCache.getUnreadMessagesByDomainId(mDomainId)));
//        } else {
//            // cache 空，直接请求服务端。不为空，没有unread，读取本地。
//            observable = mAPIRequest.getMessageList(mDomainId);
//        }
        observable.subscribe(new SingleObserver<List<AIMessage>>() {
            @Override
            public void onNext(List<AIMessage> deltaMessageList) {
                AIUtil.filterContentTypeNull(deltaMessageList);
                mMessageAdapter.setList(deltaMessageList);
                mLayoutManager.setStackFromEnd(true);
                binding.rvMessageList.scrollToPosition(deltaMessageList.size() - 1);
                mMessageAdapter.getUpFetchModule().setUpFetching(false);
                initPreSendMessage();
            }

            @Override
            public void onError(Throwable e) {
                super.onError(e);
                super.onError(e);
                mMessageAdapter.getUpFetchModule().setUpFetching(false);
            }
        });
        if (mConfig.isShowAccessible()) {
            RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) binding.rvMessageList.getLayoutParams();
            layoutParams.setMargins(0, ConvertUtils.dp2px(56), 0, ConvertUtils.dp2px(118));
            binding.rvMessageList.setLayoutParams(layoutParams);
        }
    }

    private void initPreSendMessage() {
        if (!StringUtils.isEmpty(mConfig.getPreSendMessage()) && !StringUtils.isEmpty(mConfig.getPreMessageType())) {
            if (StringUtils.equals(mConfig.getPreMessageType(), AIConst.PRE_MESSAGE_TYPE.TEXT)) {
                sendMessage(AIUtil.createTextMessage(mConfig.getPreSendMessage(), mConfig.getDomainId(), mConfig.getMetaData()));
            } else if (StringUtils.equals(mConfig.getPreMessageType(), AIConst.PRE_MESSAGE_TYPE.ANALYSIS)) {
                WorkflowRequest workflowRequest = new WorkflowRequest();
                workflowRequest.setQuestionAnalysis(AIConst.WorkFlowType.QUESTION_ANALYSIS, mConfig.getDomainId(), mConfig.getPreSendMessage());
                sendMessage(AIUtil.createQuestionAnalysisMessage(mConfig.getPreSendMessage(), mConfig.getQuestion(), mConfig.getDomainId(), mConfig.getMetaData()));
            } else if (StringUtils.equals(mConfig.getPreMessageType(), AIConst.PRE_MESSAGE_TYPE.IDEA)) {
                WorkflowRequest workflowRequest = new WorkflowRequest();
                workflowRequest.setQuestionAnalysis(AIConst.WorkFlowType.QUESTION_IDEA, mConfig.getDomainId(), mConfig.getPreSendMessage());
                sendMessage(AIUtil.createQuestionAnalysisMessage(mConfig.getPreSendMessage(), mConfig.getQuestion(), mConfig.getDomainId(), mConfig.getMetaData()));
            }
        }
    }

    private void initSpeechEngine() {
        mAPIRequest.getAsrToken().subscribe(new SingleObserver<AudioTokenData>() {
            @Override
            public void onNext(AudioTokenData audioTokenData) {
                if (StringUtils.equals(AIConst.AudioEngine.FUN, audioTokenData.getEngine())) {
                    mSpeechEngineAdapter = new FunAsrEngineAdapter(AIMessageListActivity.this);
                    binding.aiMessageBox.setSpeechEngineAdapter(mSpeechEngineAdapter);
                } else if (StringUtils.equals(AIConst.AudioEngine.VOLCENGINE, audioTokenData.getEngine())) {
                    SpeechEngineSDK.init(AIMessageListActivity.this);
                    AsrUtil.createStreamRecorder();
                    AsrUtil.createSpeechEngine();
                    AsrUtil.configInitParams();
                    AsrUtil.initEngine(binding.aiMessageBox);
                    mSpeechEngineAdapter = new BytedanceSpeechEngineAdapter();
                    binding.aiMessageBox.setSpeechEngineAdapter(mSpeechEngineAdapter);
                } else {
                    ToastUtils.showShort("未适配语音引擎");
                }
            }

            @Override
            public void onError(Throwable e) {
                ToastUtils.showShort("语音引擎信息获取失败：" + e.getMessage());
            }
        });
    }

    private void initViewModel() {
        if (mAIViewModel == null) {
            mAIViewModel = new ViewModelProvider(this).get(AIViewModel.class);
        }
        if (mObserver == null) {
            mObserver = aiSocketMessage -> {
                //收到以后
                try {
                    if (aiSocketMessage.getTotal() > 0) {
                        AIUnreadMessageCache.replace(aiSocketMessage);
                        for (AISocketMessage.SocketMessageData message : aiSocketMessage.getCounters()) {
                            if (StringUtils.equals(message.getDomainId(), mConfig.getDomainId())) {
                                mAPIRequest.getPullUnread(mConfig.getDomainId())
                                        .subscribe(new SingleObserver<List<AIMessage>>() {
                                            @Override
                                            public void onNext(List<AIMessage> list) {
                                                for (AIMessage aiMessage : list) {
                                                    aiMessage.setEvent(AIConst.SSEEventType.MESSAGE_COMPLETED);
                                                    mMessageAdapter.addData(aiMessage);
                                                }
                                                //pull 未读消息，
                                                AIUnreadMessageCache.removeUnread(message.getDomainId());
                                            }
                                        });
                            }
                        }
                    }
                } catch (Exception ex) {
                    Log.d("AIUnreadMessageCache", "getUnreadMessagesByDomainId: ");
                }
            };
        }
        mAIViewModel.getMessageLiveData().observe(this, mObserver);
    }

    private void clearViewModel() {
        if (mAIViewModel != null) {
            mAIViewModel.getMessageLiveData().removeObservers(this);
        }
        mAIViewModel = null;
        mObserver = null;
    }

    private void showFeedback(String chatId) {
        binding.includeFeedback.getRoot().setVisibility(View.VISIBLE);
        binding.includeFeedback.ivClose.setOnClickListener(v -> binding.includeFeedback.getRoot().setVisibility(View.GONE));
        binding.includeFeedback.feedbackGood.setOnClickListener(v -> {
            AIFeedback aiFeedback = new AIFeedback(chatId, "good");
            mAPIRequest.feedback(aiFeedback).subscribe();
            ToastUtils.showShort(R.string.toast_feedback);
            binding.includeFeedback.getRoot().setVisibility(View.GONE);
        });
        binding.includeFeedback.feedbackNormal.setOnClickListener(v -> {
            AIFeedback aiFeedback = new AIFeedback(chatId, "normal");
            mAPIRequest.feedback(aiFeedback).subscribe();
            ToastUtils.showShort(R.string.toast_feedback);
            binding.includeFeedback.getRoot().setVisibility(View.GONE);
        });
        binding.includeFeedback.feedbackBad.setOnClickListener(v -> {
            AIFeedback aiFeedback = new AIFeedback(chatId, "bad");
            mAPIRequest.feedback(aiFeedback).subscribe();
            ToastUtils.showShort(R.string.toast_feedback);
            binding.includeFeedback.getRoot().setVisibility(View.GONE);
        });
    }

    private void getNetMessageList(String domainId, String beforeMessageId) {
        mAPIRequest.getMessageListDelta(domainId, beforeMessageId, null)
                .subscribe(new SingleObserver<List<AIMessage>>() {
                    @Override
                    public void onNext(List<AIMessage> deltaMessageList) {
                        if (deltaMessageList != null) {
                            AIUtil.filterContentTypeNull(deltaMessageList);
                            mMessageAdapter.addData(0, deltaMessageList);
                        }
                        mMessageAdapter.getUpFetchModule().setUpFetching(false);
                    }

                    @Override
                    public void onError(Throwable e) {
                        ToastUtils.showShort("消息加载失败:" + e.getMessage());
                        mMessageAdapter.getUpFetchModule().setUpFetching(false);
                    }
                });
    }

    //生成一条用户发送的Message
    private AIMessage createUserSendMessage(String content) {
        final AIMessage userSendMessage = new AIMessage();
        userSendMessage.setRole(AIMessage.USER_ROLE);
        userSendMessage.setContent(content);
        return userSendMessage;
    }

    //模拟生成一条服务器接收到的Message，用于 loading 显示
    private AIMessage createPlaceholderMessage() {
        final AIMessage message = new AIMessage();
        message.setContent("");
        message.setRole("assistant");
        message.setEvent(AIConst.AIMessageStatus.MESSAGE_LOADING);
        return message;
    }
}
