package com.edusoho.aiassistant;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.graphics.Typeface;
import android.os.Bundle;
import android.text.SpannableString;
import android.text.Spanned;
import android.text.TextUtils;
import android.text.style.AbsoluteSizeSpan;
import android.text.style.ForegroundColorSpan;
import android.text.style.StyleSpan;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.TextView;

import androidx.activity.result.ActivityResultLauncher;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;

import com.applandeo.materialcalendarview.CalendarView;
import com.applandeo.materialcalendarview.DatePicker;
import com.applandeo.materialcalendarview.builders.DatePickerBuilder;
import com.applandeo.materialcalendarview.listeners.OnSelectDateListener;
import com.blankj.utilcode.util.ActivityUtils;
import com.blankj.utilcode.util.ColorUtils;
import com.blankj.utilcode.util.StringUtils;
import com.blankj.utilcode.util.ToastUtils;
import com.edusoho.aiassistant.databinding.ActivityLearningScheduleBinding;
import com.edusoho.aiassistant.entity.WorkflowRequest;
import com.edusoho.aiassistant.entity.WorkflowResult;
import com.edusoho.aiassistant.http.APIRequest;
import com.edusoho.aiassistant.http.SingleObserver;
import com.edusoho.aiassistant.widget.WeekDayTextView;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.TreeSet;

public class LearningScheduleActivity extends AppCompatActivity {
    private APIRequest                      mAPIRequest        = new APIRequest();
    private List<String>                    mDeadlines         = new ArrayList<>();
    private ActivityLearningScheduleBinding binding;
    private String                          mTodayDate         = new SimpleDateFormat("yyyy-MM-dd").format(Calendar.getInstance().getTime());
    private String                          mStartTime;
    private String                          mEndTime;
    private List<Integer>                   mAvailableWeekdays = new ArrayList<>();
    private TreeSet<Integer>                mResultWeekdays    = new TreeSet<>();

    public static void launch(ActivityResultLauncher<Intent> launcher) {
        Activity currActivity = ActivityUtils.getTopActivity();
        launcher.launch(new Intent(currActivity, LearningScheduleActivity.class));
    }

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        binding = ActivityLearningScheduleBinding.inflate(LayoutInflater.from(this));
        setContentView(binding.getRoot());
        binding.ivBack.setOnClickListener(v -> finish());
        initViews();
        requestData();
    }

    private void initViews() {
        renderStepTitle(getString(R.string.schedule_time_duration_text), binding.tvStepTitle1);
        renderStepTitle(getString(R.string.schedule_weekly_text), binding.tvStepTitle2);
    }

    private void requestData() {
        WorkflowRequest workflowRequest = new WorkflowRequest();
        workflowRequest.setWorkflow(AIConst.WorkFlowType.PLAN_GENERATE_CONFIG, 1271);
        mAPIRequest.getWorkflow(workflowRequest)
                .subscribe(new SingleObserver<WorkflowResult>() {
                    @Override
                    public void onNext(WorkflowResult workflowResult) {
                        mDeadlines = workflowResult.getOutputs().getDeadlines();
                        initPlanViews();
                    }

                    @Override
                    public void onError(Throwable e) {
                        super.onError(e);
                    }
                });
    }

    private void initPlanViews() {
        binding.tvStartTime.setOnClickListener(v -> showDateDialog(getStartDateListener()));
        binding.tvEndTime.setOnClickListener(v -> {
            if (mDeadlines.isEmpty()) {
                showDateDialog(getEndDateListener());
            } else {
                showDeadlinesDialog();
            }
        });

        binding.tvNext.setOnClickListener(v -> {
            if (isStartTimeLaterThanEndTime(mStartTime, mEndTime)) {
                ToastUtils.showLong("开始时间不能晚于结束时间");
            } else {
                binding.llDateSelect.setVisibility(View.GONE);
                binding.llWeekdaySelect.setVisibility(View.VISIBLE);
                setWeekdays(mStartTime, mEndTime);
                renderWeekdays(mAvailableWeekdays);
            }
        });
        binding.tvPrev.setOnClickListener(v -> {
            binding.llDateSelect.setVisibility(View.VISIBLE);
            binding.llWeekdaySelect.setVisibility(View.GONE);
        });
        for (int i = 0; i < binding.clWeekdays.getChildCount(); i++) {
            WeekDayTextView weekDayTextView = (WeekDayTextView) binding.clWeekdays.getChildAt(i);
            if (weekDayTextView.getTag() != null) {
                weekDayTextView.setOnClickListener(v -> {
                    if (v.isEnabled()) {
                        if (v.isSelected()) {
                            v.setSelected(false);
                            mResultWeekdays.remove(Integer.parseInt(v.getTag().toString()));
                        } else {
                            v.setSelected(true);
                            mResultWeekdays.add(Integer.parseInt(v.getTag().toString()));
                        }
                    }
                    checkCreateButton();
                });
            }
        }
        binding.tvCreate.setOnClickListener(v -> {
            Intent intent = new Intent()
                    .putExtra("startDate", mStartTime)
                    .putExtra("endDate", mEndTime)
                    .putExtra("weekDays", mResultWeekdays);
            setResult(Activity.RESULT_OK, intent);
            finish();
        });
    }

    private void renderStepTitle(String title, TextView textView) {
        StringBuffer stringBuffer = new StringBuffer(title);
        SpannableString spannableString = new SpannableString(stringBuffer);
        spannableString.setSpan(new StyleSpan(Typeface.BOLD), 0, 1, Spanned.SPAN_INCLUSIVE_INCLUSIVE);
        spannableString.setSpan(new AbsoluteSizeSpan(24, true), 0, 1, Spanned.SPAN_INCLUSIVE_INCLUSIVE);
        spannableString.setSpan(new ForegroundColorSpan(ColorUtils.getColor(R.color.color_165DFF)), 0, 1, Spanned.SPAN_INCLUSIVE_INCLUSIVE);
        textView.setText(spannableString);
    }

    private void checkCreateButton() {
        binding.tvCreate.setEnabled(!mResultWeekdays.isEmpty());
    }

    private void renderWeekdays(List<Integer> availableWeekdays) {
        for (int i = 0; i < binding.clWeekdays.getChildCount(); i++) {
            WeekDayTextView weekDayTextView = (WeekDayTextView) binding.clWeekdays.getChildAt(i);
            if (weekDayTextView.getTag() != null) {
                if (availableWeekdays.contains(Integer.parseInt(weekDayTextView.getTag().toString()))) {
                    weekDayTextView.setEnabled(true);
                } else {
                    weekDayTextView.setEnabled(false);
                    mResultWeekdays.remove(Integer.parseInt(weekDayTextView.getTag().toString()));
                }
            }
        }
        checkCreateButton();
    }

    private boolean isStartTimeLaterThanEndTime(String startTime, String endTime) {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            Date start = sdf.parse(startTime);
            Date end = sdf.parse(endTime);
            return start.after(end);
        } catch (ParseException e) {
            e.printStackTrace();
            return false;
        }
    }

    private boolean isNextEnabled() {
        return !StringUtils.isEmpty(binding.tvStartTime.getText().toString()) && !StringUtils.isEmpty(binding.tvEndTime.getText().toString());
    }

    private void showDateDialog(OnSelectDateListener listener) {
        Calendar minDate = Calendar.getInstance();
        minDate.set(Calendar.HOUR_OF_DAY, 0);
        minDate.set(Calendar.MINUTE, 0);
        minDate.set(Calendar.SECOND, 0);
        minDate.set(Calendar.MILLISECOND, 0);

        Calendar maxDate = Calendar.getInstance();
        maxDate.set(Calendar.YEAR, 2100);
        maxDate.set(Calendar.MONTH, Calendar.DECEMBER);
        maxDate.set(Calendar.DAY_OF_MONTH, 31);

        DatePickerBuilder builder = new DatePickerBuilder(LearningScheduleActivity.this, listener)
                .pickerType(CalendarView.ONE_DAY_PICKER)
                .minimumDate(minDate)
                .maximumDate(maxDate)
                .date(getSelectedDate(mTodayDate));
        DatePicker datePicker = builder.build();
        datePicker.show();
    }

    private OnSelectDateListener getStartDateListener() {
        return list -> {
            if (!list.isEmpty()) {
                Calendar selectedDate = list.get(0);
                String selectedDateStr = new SimpleDateFormat("yyyy-MM-dd").format(selectedDate.getTime());
                binding.tvStartTime.setText(selectedDateStr);
                mStartTime = selectedDateStr;
                binding.tvNext.setEnabled(isNextEnabled());
            }
        };
    }

    private OnSelectDateListener getEndDateListener() {
        return list -> {
            Calendar selectedDate = list.get(0);
            String selectedDateStr = new SimpleDateFormat("yyyy-MM-dd").format(selectedDate.getTime());
            binding.tvEndTime.setText(selectedDateStr);
            mEndTime = selectedDateStr;
            binding.tvNext.setEnabled(isNextEnabled());
        };
    }

    private void showDeadlinesDialog() {
        AlertDialog.Builder builder = new AlertDialog.Builder(LearningScheduleActivity.this);
        builder.setTitle("选择截止日期");

        // 将截止日期列表转换为 String 数组
        String[] deadlineArray = mDeadlines.toArray(new String[0]);

        // 设置单选列表项
        builder.setSingleChoiceItems(deadlineArray, -1, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                binding.tvEndTime.setText(deadlineArray[which]);
                mEndTime = deadlineArray[which];
                binding.tvNext.setEnabled(isNextEnabled());
                dialog.dismiss();
            }
        });

        // 设置取消按钮
        builder.setNegativeButton("取消", null);

        AlertDialog dialog = builder.create();
        dialog.show();
    }

    private Calendar getSelectedDate(String selectedDate) {
        if (!TextUtils.isEmpty(selectedDate)) {
            try {
                Date date = new SimpleDateFormat("yyyy-MM-dd").parse(selectedDate);
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(date);
                return calendar;
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    public void setWeekdays(String startDateStr, String endDateStr) {
        // 清空之前的星期数据
        mAvailableWeekdays.clear();

        // 定义日期格式
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        try {
            // 解析开始和结束日期
            Date startDate = sdf.parse(startDateStr);
            Date endDate = sdf.parse(endDateStr);

            // 使用 Calendar 处理日期
            Calendar startCal = Calendar.getInstance();
            startCal.setTime(startDate);
            Calendar endCal = Calendar.getInstance();
            endCal.setTime(endDate);

            // 使用 TreeSet 去重并自动排序
            TreeSet<Integer> weekdaySet = new TreeSet<>();

            // 从开始日期到结束日期遍历
            while (!startCal.getTime().after(endCal.getTime())) {
                // 获取星期几（1=星期日, 2=星期一, ..., 7=星期六）
                int dayOfWeek = startCal.get(Calendar.DAY_OF_WEEK);
                // 转换为 1=星期一, 7=星期日
                int weekday = (dayOfWeek + 5) % 7 + 1;
                weekdaySet.add(weekday);

                // 日期加一天
                startCal.add(Calendar.DAY_OF_MONTH, 1);
            }

            // 将排序后的结果放入 mAvailableWeekdays
            mAvailableWeekdays.addAll(weekdaySet);

        } catch (ParseException e) {
            e.printStackTrace();
        }
    }
}
