package com.edusoho.aiassistant.speech;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.media.AudioFormat;
import android.media.AudioRecord;
import android.media.MediaRecorder;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresPermission;
import androidx.core.app.ActivityCompat;

import com.blankj.utilcode.constant.PermissionConstants;
import com.blankj.utilcode.util.ActivityUtils;
import com.blankj.utilcode.util.PermissionUtils;
import com.blankj.utilcode.util.ToastUtils;
import com.bytedance.speech.speechengine.SpeechEngine;
import com.edusoho.aiassistant.http.APIRequest;
import com.edusoho.aiassistant.entity.AudioTokenData;
import com.edusoho.aiassistant.http.SingleObserver;
import com.edusoho.aiassistant.util.SSLSocketClient;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import okio.ByteString;

public class FunAsrEngineAdapter implements SpeechEngineAdapter {
    private static final String TAG            = "FunAsrEngineAdapter";
    private static final String DEFAULT_HOST   = "wss://funasr.edusoho.cn?token=";
    private static final String MODE           = "2pass";
    private static final String CHUNK_SIZE     = "5, 10, 5";
    private static final int    CHUNK_INTERVAL = 10;
    private static final int    SEND_SIZE      = 1920;
    private static final int    SAMPLE_RATE    = 16000;
    private static final int    CHANNEL        = AudioFormat.CHANNEL_IN_MONO;
    private static final int    AUDIO_FORMAT   = AudioFormat.ENCODING_PCM_16BIT;

    private       AudioRecord       audioRecord;
    private       boolean           isRecording = false;
    private       String            ASR_HOST    = "";
    private       String            hotWords    = "阿里巴巴 20\n达摩院 20\n夜雨飘零 20\n";
    private       WebSocket         webSocket;
    private       String            allAsrText  = "";
    private       String            asrText     = "";
    private       AsrResultListener asrResultListener;
    private final APIRequest        apiRequest  = new APIRequest();
    private       long              mAudioStartTime;
    private       long              mAudioDuration;

    public FunAsrEngineAdapter(Context context) {
        // 获取 token
        apiRequest.getAsrToken().subscribe(new SingleObserver<AudioTokenData>() {
            @Override
            public void onNext(AudioTokenData tokenData) {
                if (tokenData != null) {
                    ASR_HOST = String.format("%s?token=%s", tokenData.getServer(), tokenData.getToken());
                }
            }
        });
    }

    @Override
    public void startRecording(Context context, Runnable onError) {
        isRecording = true;
        try {
            PermissionUtils.permissionGroup(PermissionConstants.MICROPHONE, PermissionConstants.STORAGE)
                    .callback(new PermissionUtils.FullCallback() {
                        @RequiresPermission(Manifest.permission.RECORD_AUDIO)
                        @Override
                        public void onGranted(@NonNull List<String> granted) {
                            allAsrText = "";
                            asrText = "";
                            audioRecord = new AudioRecord(MediaRecorder.AudioSource.MIC, SAMPLE_RATE, CHANNEL, AUDIO_FORMAT, SEND_SIZE);
                            Thread recordingAudioThread = new Thread(() -> {
                                try {
                                    setAudioData();
                                } catch (Exception e) {
                                    Log.e(TAG, "Error in setAudioData", e);
                                    if (onError != null) {
                                        ActivityUtils.getTopActivity().runOnUiThread(onError);
                                    }
                                }
                            });
                            mAudioStartTime = System.currentTimeMillis();
                            recordingAudioThread.start();
                            audioRecord.startRecording();
                        }

                        @Override
                        public void onDenied(@NonNull List<String> deniedForever, @NonNull List<String> denied) {
                            if (onError != null) {
                                onError.run();
                            }
                        }
                    })
                    .request();
        } catch (IllegalStateException e) {
            Log.e(TAG, "Failed to initialize AudioRecord", e);
            if (onError != null) {
                onError.run();
            }
        }
    }

    @Override
    public void stopRecording(boolean isSend) {
        isRecording = false;
        if (audioRecord != null) {
            audioRecord.stop();
            audioRecord.release();
            audioRecord = null;
        }
    }

    @Override
    public SpeechEngine getSpeechEngine() {
        return null; // FunASR 使用 WebSocket，不依赖 SpeechEngine
    }

    @Override
    public void setAsrResultListener(AsrResultListener listener) {
        this.asrResultListener = listener;
    }

    private void setAudioData() throws Exception {
        // 建立WebSocket连接
        OkHttpClient client = new OkHttpClient.Builder()
                // 忽略验证证书
                .sslSocketFactory(SSLSocketClient.getSSLSocketFactory(), SSLSocketClient.getX509TrustManager())
                // 不验证域名
                .hostnameVerifier(SSLSocketClient.getHostnameVerifier())
                .build();
        Request request = new Request.Builder()
                .url(ASR_HOST)
                .build();
        webSocket = client.newWebSocket(request, new WebSocketListener() {
            @Override
            public void onOpen(@NonNull WebSocket webSocket, @NonNull Response response) {
                Log.d(TAG, "WebSocket连接成功");
            }

            @Override
            public void onMessage(@NonNull WebSocket webSocket, @NonNull String text) {
                Log.d(TAG, "WebSocket接收到消息: " + text);
                try {
                    JSONObject jsonObject = new JSONObject(text);
                    String t = jsonObject.getString("text");
                    boolean isFinal = jsonObject.getBoolean("is_final");
                    if (!t.isEmpty()) {
                        String mode = jsonObject.getString("mode");
                        if (mode.equals("2pass-offline")) {
                            asrText = "";
                            allAsrText += t;
                        } else {
                            asrText += t;
                        }
                        Log.d(TAG, "语音【连续】识别结果: " + allAsrText + asrText);
                        if (isFinal && asrResultListener != null) {
                            asrResultListener.onAsrResult(allAsrText + asrText, mAudioDuration);
                            Log.d(TAG, "语音【最终】识别结果: " + allAsrText + asrText);
                        }
                    }
                    if (isFinal) {
                        webSocket.close(1000, "关闭WebSocket连接");
                    }
                } catch (JSONException e) {
                    Log.e(TAG, "Error parsing message", e);
                }
            }

            @Override
            public void onClosing(@NonNull WebSocket webSocket, int code, @NonNull String reason) {
                Log.d(TAG, "WebSocket关闭连接: " + reason);
            }

            @Override
            public void onFailure(@NonNull WebSocket webSocket, @NonNull Throwable t, Response response) {
                Log.d(TAG, "WebSocket连接失败: " + t.getMessage());
            }
        });

        String message = getMessage(true);
        Log.d(TAG, "WebSocket发送消息: " + message);
        webSocket.send(message);
        audioRecord.startRecording();
        byte[] bytes = new byte[SEND_SIZE];
        while (isRecording && audioRecord != null) {
            int readSize = audioRecord.read(bytes, 0, SEND_SIZE);
            if (readSize > 0) {
                ByteString byteString = ByteString.of(bytes);
                webSocket.send(byteString);
            }
        }
        mAudioDuration = System.currentTimeMillis() - mAudioStartTime;
        JSONObject obj = new JSONObject();
        obj.put("is_speaking", false);
        webSocket.send(obj.toString());
    }

    private String getMessage(boolean isSpeaking) {
        try {
            JSONObject obj = new JSONObject();
            obj.put("mode", MODE);
            JSONArray array = new JSONArray();
            String[] chunkList = CHUNK_SIZE.split(",");
            for (String s : chunkList) {
                array.put(Integer.valueOf(s.trim()));
            }
            obj.put("chunk_size", array);
            obj.put("chunk_interval", CHUNK_INTERVAL);
            obj.put("wav_name", "default");
            if (!hotWords.isEmpty()) {
                JSONObject hotwordsJSON = new JSONObject();
                String[] hotWordsList = hotWords.split("\n");
                for (String s : hotWordsList) {
                    if (s.isEmpty()) {
                        Log.w(TAG, "hotWords为空");
                        continue;
                    }
                    String[] hotWordsArray = s.split(" ");
                    if (hotWordsArray.length != 2) {
                        Log.w(TAG, "hotWords格式不正确");
                        continue;
                    }
                    hotwordsJSON.put(hotWordsArray[0], Integer.valueOf(hotWordsArray[1]));
                }
                obj.put("hotwords", hotwordsJSON.toString());
            }
            obj.put("wav_format", "pcm");
            obj.put("is_speaking", isSpeaking);
            return obj.toString();
        } catch (Exception e) {
            Log.e(TAG, "Error creating message", e);
        }
        return "";
    }
}