package com.edusoho.aiassistant.widget;

import android.Manifest;
import android.content.Context;
import android.os.Handler;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.AttributeSet;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.RelativeLayout;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.LinearLayoutCompat;

import com.blankj.utilcode.util.ActivityUtils;
import com.blankj.utilcode.util.ColorUtils;
import com.blankj.utilcode.util.ConvertUtils;
import com.blankj.utilcode.util.KeyboardUtils;
import com.blankj.utilcode.util.PermissionUtils;
import com.blankj.utilcode.util.ToastUtils;
import com.bytedance.speech.speechengine.SpeechEngine;
import com.bytedance.speech.speechengine.SpeechEngineDefines;
import com.edusoho.aiassistant.R;
import com.edusoho.aiassistant.databinding.AiMessageDialogBoxBinding;
import com.edusoho.aiassistant.http.SSEClient;
import com.edusoho.aiassistant.speech.AsrUtil;
import com.edusoho.aiassistant.speech.BytedanceSpeechEngineAdapter;
import com.edusoho.aiassistant.speech.FunAsrEngineAdapter;
import com.edusoho.aiassistant.speech.SpeechEngineAdapter;
import com.edusoho.aiassistant.util.AIUtil;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.List;

public class AIMessageDialogBox extends RelativeLayout implements SpeechEngine.SpeechListener {
    private static final String                    TAG                     = "AIMessageDialogBox";
    private              AiMessageDialogBoxBinding binding;
    private              AIMessageBoxListener      listener;
    private              SSEClient                 mCurSSEClient;
    private              long                      mFinishTalkingTimestamp = -1;

    // Record
    private Handler  recordHandler   = null;
    private Runnable recordRunnable  = null;
    private boolean  recordIsRunning = false;

    private SpeechEngineAdapter speechEngineAdapter; // 语音引擎适配器

    public AIMessageDialogBox(Context context) {
        super(context);
        init(context);
    }

    public AIMessageDialogBox(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        init(context);
    }

    public AIMessageDialogBox(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context);
    }

    public void setListener(AIMessageBoxListener listener) {
        this.listener = listener;
    }

    public void setCurrentSSEClient(SSEClient sseClient) {
        this.mCurSSEClient = sseClient;
    }

    public void setSpeechEngineAdapter(SpeechEngineAdapter adapter) {
        speechEngineAdapter = adapter;
        speechEngineAdapter.setAsrResultListener((text, audioDuration) -> {
            if (listener != null) listener.sendText(text, true, audioDuration);
        });
    }

    private void init(Context context) {
        binding = AiMessageDialogBoxBinding.inflate(LayoutInflater.from(context), this);
        // 默认使用 Bytedance 语音引擎
        binding.etMessageText.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {

            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {

            }

            @Override
            public void afterTextChanged(Editable s) {
                setTextInputMode();
            }
        });
        binding.ivRadio.setOnClickListener(v -> setAudioInputMode());
        binding.ivKeyboard.setOnClickListener(v -> {
            setTextInputMode();
            KeyboardUtils.showSoftInput(this);
            binding.etMessageText.requestFocus();
        });

        binding.ivAudioAnimation.setPadding(0, -120, 0, -120);

        binding.tvSpeakButton.setVoiceButtonListener(new AISpeakButton.VoiceButtonListener() {
            @Override
            public void onRecordStart() {
                Log.d("tvSpeakButton", "onRecordStart: ");
                binding.rlInputBox.setBackgroundResource(R.drawable.bg_input_message_box_record);
                binding.rlFunctionPanel.setVisibility(View.GONE);
                binding.ivAudioAnimation.setVisibility(View.VISIBLE);
                binding.tvSpeakButton.setVisibility(GONE);
                binding.llConvertTextBox.setVisibility(VISIBLE);
                binding.ivAudioAnimation.playAnimation();
                LayoutParams layoutParams = (LayoutParams) getLayoutParams();
                layoutParams.height = ConvertUtils.dp2px(180);
                setLayoutParams(layoutParams);
                setBackgroundColor(ColorUtils.getColor(android.R.color.white));
                recordBtnTouchDown();
            }

            @Override
            public void onRecordSlide(boolean isSlided) {
                Log.d("tvSpeakButton", "onRecordSlide: " + isSlided);
                if (isSlided) {
                    binding.tvPressHint.setText(R.string.press_up);
                    binding.rlInputBox.setBackgroundResource(R.drawable.bg_input_message_box_slided);
                } else {
                    binding.tvPressHint.setText(R.string.press_down);
                    binding.rlInputBox.setBackgroundResource(R.drawable.bg_input_message_box_record);
                }
            }

            @Override
            public void onRecordStop(boolean isRecord) {
                if (isRecord) {
                    Log.d("tvSpeakButton", "onRecordStop: 发送录音记录");
                    speechEngineAdapter.stopRecording(true);
//                    recordBtnTouchUp(true);
                } else {
                    speechEngineAdapter.stopRecording(false);
//                    recordBtnTouchUp(false);
                    Log.d("tvSpeakButton", "onRecordStop: 不发送");
                }
                resetUI();
            }
        });

        binding.ivSendText.setOnClickListener(l -> {
            if (listener != null) {
                listener.sendText(binding.etMessageText.getText().toString(), false, 0);// 触发 setAnswerGeneratedMode();
                binding.etMessageText.setText(""); //触发 afterText
                KeyboardUtils.hideSoftInput(this);
            }
        });
    }

    //文字输入状态
    public void setTextInputMode() {
        if (binding.etMessageText.getText() != null && binding.etMessageText.getText().toString().isEmpty()) {
            binding.ivRadio.setVisibility(View.VISIBLE);
            binding.ivSendText.setVisibility(View.GONE);
        } else {
            binding.ivRadio.setVisibility(View.GONE);
            binding.ivSendText.setVisibility(View.VISIBLE);
        }
        binding.ivKeyboard.setVisibility(View.GONE);
        binding.etMessageText.setVisibility(View.VISIBLE);
        binding.tvSpeakButton.setVisibility(View.GONE);
    }

    //语音输入状态
    public void setAudioInputMode() {
        binding.ivRadio.setVisibility(View.GONE);
        binding.ivSendText.setVisibility(View.GONE);
        binding.ivKeyboard.setVisibility(View.VISIBLE);
        binding.etMessageText.setVisibility(View.GONE);
        binding.tvSpeakButton.setVisibility(View.VISIBLE);
        KeyboardUtils.hideSoftInput(this);
    }

    public void stopAnswer() {
        // 由于 cancel sse 是异步，更改 MESSAGE 状态会有延迟，
        // 因此主动是更改上一条 Message 状态，而不是等待 sse 的 onFailure
        if (mCurSSEClient != null) {
            mCurSSEClient.cancel();
        }
    }

    private void resetUI() {
        binding.rlInputBox.setBackgroundResource(R.drawable.bg_input_message_box_normal);
        binding.rlFunctionPanel.setVisibility(View.VISIBLE);
        binding.ivAudioAnimation.setVisibility(View.GONE);
        binding.tvSpeakButton.setVisibility(View.VISIBLE);
        binding.ivAudioAnimation.cancelAnimation();
        binding.llConvertTextBox.setVisibility(GONE);
        LayoutParams layoutParams = (LayoutParams) getLayoutParams();
        layoutParams.height = ConvertUtils.dp2px(80);
        setLayoutParams(layoutParams);
        setBackgroundColor(ColorUtils.getColor(android.R.color.transparent));
    }

    private void recordBtnTouchDown() {
        recordIsRunning = false;
        recordHandler = new Handler();
        recordRunnable = () -> {
            recordIsRunning = true;
            speechEngineAdapter.startRecording(getContext(), () -> {
                // 错误处理回调
                recordIsRunning = false;
                ActivityUtils.getTopActivity().runOnUiThread(() -> {
                    ToastUtils.showShort("录音启动失败");
                    resetUI();
                });
            });
        };
        recordHandler.postDelayed(recordRunnable, 500);
    }

    private void recordBtnTouchUp(boolean isSend) {
        if (recordIsRunning) {
            recordIsRunning = false;
            Log.i(TAG, "AsrTouch: Finish");
            mFinishTalkingTimestamp = System.currentTimeMillis();
            // Directive：结束用户音频输入。
            Log.i(TAG, "Directive: DIRECTIVE_FINISH_TALKING");
            AsrUtil.getSpeechEngine().sendDirective(SpeechEngineDefines.DIRECTIVE_FINISH_TALKING, "");
            if (isSend) {
                AsrUtil.getSpeechEngine().sendDirective(SpeechEngineDefines.DIRECTIVE_FINISH_TALKING, "");
            } else {
                AsrUtil.getSpeechEngine().sendDirective(SpeechEngineDefines.DIRECTIVE_SYNC_STOP_ENGINE, "");
            }
            AsrUtil.getStreamRecorder().Stop();
        } else if (recordRunnable != null) {
            Log.i(TAG, "AsrTouch: Cancel");
            recordHandler.removeCallbacks(recordRunnable);
            recordRunnable = null;
        }
    }

    public void speechAsrResult(final String data, boolean isFinal) {
        // 计算由录音结束到 ASR 最终结果之间的延迟
        long delay = 0;
        if (isFinal && mFinishTalkingTimestamp > 0) {
            delay = System.currentTimeMillis() - mFinishTalkingTimestamp;
            mFinishTalkingTimestamp = 0;
        }
        final long response_delay = delay;
        ActivityUtils.getTopActivity().runOnUiThread(() -> {
            try {
                // 从回调的 json 数据中解析 ASR 结果
                JSONObject reader = new JSONObject(data);
                if (!reader.has("result")) {
                    return;
                }
                String text = reader.getJSONObject("result").getString("text");
                if (text.isEmpty()) {
                    return;
                }
                long audioDuration = reader.getJSONObject("audio_info").getLong("duration");
                if (isFinal) {
                    if (listener != null) {
                        listener.sendText(text, true, audioDuration);
                    }
                }
                setResultText(text);
            } catch (JSONException e) {
                ToastUtils.showShort(e.getMessage());
            }
        });
    }

    private void setResultText(String text) {
        Log.d(TAG, "setResultText: " + text);
    }

    @Override
    public void onSpeechMessage(int type, byte[] data, int len) {
        String stdData = new String(data);
        switch (type) {
            case SpeechEngineDefines.MESSAGE_TYPE_ENGINE_START:
                // Callback: 引擎启动成功回调
                Log.i(TAG, "Callback: 引擎启动成功: data: " + stdData);
//                speechStart();
                break;
            case SpeechEngineDefines.MESSAGE_TYPE_ENGINE_STOP:
                // Callback: 引擎关闭回调
                Log.i(TAG, "Callback: 引擎关闭: data: " + stdData);
//                speechStop();
                break;
            case SpeechEngineDefines.MESSAGE_TYPE_ENGINE_ERROR:
                // Callback: 错误信息回调
                Log.e(TAG, "Callback: 错误信息: " + stdData);
//                speechError(stdData);
                break;
            case SpeechEngineDefines.MESSAGE_TYPE_CONNECTION_CONNECTED:
                Log.i(TAG, "Callback: 建连成功: data: " + stdData);
                break;
            case SpeechEngineDefines.MESSAGE_TYPE_PARTIAL_RESULT:
                // Callback: ASR 当前请求的部分结果回调
                Log.d(TAG, "Callback: ASR 当前请求的部分结果");
//                speechAsrResult(stdData, false);
                break;
            case SpeechEngineDefines.MESSAGE_TYPE_FINAL_RESULT:
                // Callback: ASR 当前请求最终结果回调
                Log.i(TAG, "Callback: ASR 当前请求最终结果");
                speechAsrResult(stdData, true);
                break;
            case SpeechEngineDefines.MESSAGE_TYPE_VOLUME_LEVEL:
                // Callback: 录音音量回调
                Log.d(TAG, "Callback: 录音音量");
                break;
            default:
                break;
        }
    }

    public interface AIMessageBoxListener {
        void sendText(String msg, boolean audio, long audioDuration);
    }
}
